package com.help.autoconfig;

import com.help.SpringBootVFS;
import com.help.SpringScanConfig;
import com.help.datasource.HelpDynamicDataSource;
import com.help.datasource.HelpDynamicTransactionFactory;
import com.help.datasource.config.HelpMybatisConfig;
import com.help.common.util.StringUtil;
import com.help.dao.batch.BatchHelper;
import com.help.datasource.config.MyBatisSpringSupportConfig;
import org.apache.ibatis.annotations.Mapper;
import org.apache.ibatis.logging.slf4j.Slf4jImpl;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.session.SqlSessionFactory;
import org.apache.ibatis.transaction.TransactionFactory;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.mybatis.spring.SqlSessionTemplate;
import org.mybatis.spring.mapper.MapperScannerConfigurer;
import org.mybatis.spring.transaction.SpringManagedTransactionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;

import javax.sql.DataSource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * HELP平台MyBatis组件自动配置工具
 */
@ConditionalOnBean(value = DataSource.class, name = "defaultDataSource")
@ConditionalOnClass({MapperScannerConfigurer.class, SqlSessionFactory.class})
@AutoConfigureAfter({HelpDataSourceAutoConfiguration.class, HelpMyBatisConfigAutoConfiguration.class})
public class HelpMyBatisAutoConfiguration {

    Logger logger = LoggerFactory.getLogger(HelpMyBatisAutoConfiguration.class);

    //注入MyBatis的MapperScanner
    @Bean
    @ConditionalOnMissingBean
    public MapperScannerConfigurer mapperScannerConfigurer(SpringScanConfig springScanConfig) {
        MapperScannerConfigurer mapperScannerConfigurer = new MapperScannerConfigurer();
        mapperScannerConfigurer.setBasePackage(StringUtil.join(springScanConfig.getBasePackage(), ","));
        mapperScannerConfigurer.setAnnotationClass(Mapper.class);
        mapperScannerConfigurer.setSqlSessionTemplateBeanName("defaultSqlSessionTemplate");

        logger.info("检测到MyBatis环境,自动配置[MyBatis-Mapper扫描器],基础包目录:[" + StringUtil.join(springScanConfig.getBasePackage(), ",") + "]");

        return mapperScannerConfigurer;
    }

    //注入mybatis事务管理器,避免多数据源下的事务互串
    @Bean
    @ConditionalOnMissingBean
    public TransactionFactory transactionFactory(@Qualifier("defaultDataSource") DataSource dataSource) {
        if (dataSource instanceof HelpDynamicDataSource) {
            logger.info("检测到多数据源环境,自动配置[HELP-Mybatis动态事务管理器]");
            return new HelpDynamicTransactionFactory();
        } else {
            logger.info("检测到单数据源环境,自动配置[Spring-MyBatis默认事务管理器]");
            return new SpringManagedTransactionFactory();
        }
    }

    //注入mybatis的SqlSessionFactory
    @Bean(name = "defaultSqlSessionFactory")
    @ConditionalOnMissingBean(name = "defaultSqlSessionFactory")
    public SqlSessionFactory sqlSessionFactory(@Autowired SpringScanConfig springScanConfig, @Qualifier("defaultDataSource") DataSource dataSource, HelpMybatisConfig helpMybatisConfig, MyBatisSpringSupportConfig myBatisSpringSupportConfig, @Autowired(required = false) List<Interceptor> interceptors, @Autowired(required = false) TransactionFactory transactionFactory) {
        org.apache.ibatis.session.Configuration conf = new org.apache.ibatis.session.Configuration();
        conf.setMapUnderscoreToCamelCase(true);
        conf.setLogImpl(Slf4jImpl.class);

        SqlSessionFactoryBean bean = new SqlSessionFactoryBean();
        bean.setDataSource(dataSource);
        bean.setConfiguration(conf);
        bean.setTransactionFactory(transactionFactory);
        bean.setVfs(SpringBootVFS.class);

        if (myBatisSpringSupportConfig.getTypeAliasesSuperType() != null) {
            bean.setTypeAliasesSuperType(myBatisSpringSupportConfig.getTypeAliasesSuperType());
        }

        if (myBatisSpringSupportConfig.getTypeAliasesPackage() != null) {
            bean.setTypeAliasesPackage(myBatisSpringSupportConfig.getTypeAliasesPackage());
        }

        //添加插件
        if (interceptors != null) {
            bean.setPlugins(interceptors.toArray(new Interceptor[0]));
        }

        try {
            List<String> mapperLocation = new ArrayList<>();
            if (helpMybatisConfig.getMapperLocation() != null) {
                mapperLocation.addAll(Arrays.asList(helpMybatisConfig.getMapperLocation()));
            }
            if (myBatisSpringSupportConfig.getMapperLocations() != null) {
                mapperLocation.addAll(Arrays.asList(myBatisSpringSupportConfig.getMapperLocations()));
            }
            mapperLocation.addAll(Arrays.stream(springScanConfig.getBasePackage()).map(p -> "classpath*:" + p.replace(".", "/") + "/**/*.xml").collect(Collectors.toList()));
            mapperLocation = mapperLocation.stream().distinct().collect(Collectors.toList());

            logger.info("检测到MyBatis环境,自动配置[MyBatis-XML扫描器],xml文件所在路径[" + StringUtil.join(mapperLocation, ",") + "]");

            //添加XML目录
            ResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();

            List<Resource> resources = new ArrayList<>();
            for (String s : mapperLocation) {
                Resource[] all = resolver.getResources(s);
                resources.addAll(Arrays.asList(all));
            }

            bean.setMapperLocations(resources.toArray(new Resource[0]));

            logger.info("检测到MyBatis环境,自动配置[SqlSessionFactory]");

            return bean.getObject();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    //注入MyBatis的SqlSessionTemplate
    @Bean(name = "defaultSqlSessionTemplate")
    @ConditionalOnMissingBean(name = "defaultSqlSessionTemplate")
    public SqlSessionTemplate sqlSessionTemplate(@Qualifier("defaultSqlSessionFactory") SqlSessionFactory sqlSessionFactory) {

        logger.info("检测到MyBatis环境,自动配置[SqlSessionTemplate]");

        return new SqlSessionTemplate(sqlSessionFactory);
    }

    @Bean
    @ConditionalOnBean(SqlSessionFactory.class)
    @ConditionalOnMissingBean
    public BatchHelper batchHelper(@Qualifier("defaultSqlSessionFactory") SqlSessionFactory sqlSessionFactory) {

        logger.info("检测到MyBatis环境,自动配置[HELP平台MyBatis批处理支持组件]");
        return new BatchHelper(sqlSessionFactory);
    }
}
