package com.help.autoconfig;

import com.help.common.exception.UnifyErrorCode;
import com.help.common.exception.UnifyException;
import com.help.datasource.DefaultDataSourceBuilder;
import com.help.datasource.HelpDynamicDataSource;
import com.help.datasource.IDataSourceBuilder;
import com.help.datasource.config.HelpDataSourceConfig;
import com.help.datasource.config.HelpDataSourceConfigGroup;
import com.help.common.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.*;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import javax.sql.DataSource;
import java.util.*;

/**
 * 默认数据源自动配置组件
 */
@ConditionalOnClass(HelpDataSourceConfigGroup.class)
@ConditionalOnMissingBean(value = DataSource.class, name = "defaultDataSource")
@ConditionalOnExpression("!'${help.datasource.multi[0].url:null}'.equalsIgnoreCase('null') || !'${spring.datasource.url:null}'.equalsIgnoreCase('null')")
public class HelpDataSourceAutoConfiguration {

    Logger logger = LoggerFactory.getLogger(HelpDataSourceAutoConfiguration.class);

    //注入spring.datasource数据源配置
    @Bean
    @ConditionalOnMissingBean
    @ConditionalOnProperty(prefix = "spring.datasource", name = {"url", "driver-class-name", "username", "password"})
    @ConfigurationProperties("spring.datasource")
    public HelpDataSourceConfig defaultSpringDataSourceConfig() {
        return new HelpDataSourceConfig();
    }

    //注入help.datasource数据源配置
    @Bean
    @ConditionalOnMissingBean
    @ConfigurationProperties("help.datasource")
    public HelpDataSourceConfigGroup helpDataSourceConfigGroup() {
        return new HelpDataSourceConfigGroup();
    }

    //注入HelpDynamicDataSource动态多数据源
    @Bean("defaultDataSource")
    @ConditionalOnMissingBean(DataSource.class)
    @ConditionalOnBean(HelpDataSourceConfigGroup.class)
    @ConditionalOnExpression("!'${help.datasource.multi[1].url:null}'.equalsIgnoreCase('null') || !'${help.datasource.multi[0].url:null}'.equalsIgnoreCase('null') && !'${spring.datasource.url:null}'.equalsIgnoreCase('null')")
    public HelpDynamicDataSource helpDynamicDataSource(HelpDataSourceConfigGroup config, @Autowired(required = false) HelpDataSourceConfig defaultSpringDataSourceConfig,
                                                       @Autowired(required = false) List<IDataSourceBuilder> iDataSourceBuilderList
    ) {
        buildConfigGroup(config, defaultSpringDataSourceConfig);

        Map<String, DataSource> dataSources = buildDataSource(config, iDataSourceBuilderList);

        //多数据源环境

        HelpDynamicDataSource dynamicDataSource = new HelpDynamicDataSource();

        DataSource dft = null;

        Map<Object, Object> targetDataSource = new HashMap<>(dataSources.size());
        for (String key : dataSources.keySet()) {
            if (dft == null) {
                dft = dataSources.get(key);
            }
            targetDataSource.put(key, dataSources.get(key));
        }

        dynamicDataSource.setTargetDataSources(targetDataSource);
        dynamicDataSource.setDefaultTargetDataSource(dft);

        return dynamicDataSource;
    }

    //注入单数据源
    @Bean("defaultDataSource")
    @ConditionalOnMissingBean({DataSource.class, HelpDynamicDataSource.class})
    @ConditionalOnBean(HelpDataSourceConfigGroup.class)
    public DataSource defaultDataSource(HelpDataSourceConfigGroup config, @Autowired(required = false) HelpDataSourceConfig defaultSpringDataSourceConfig,
                                        @Autowired(required = false) List<IDataSourceBuilder> iDataSourceBuilderList
    ) {
        buildConfigGroup(config, defaultSpringDataSourceConfig);

        Map<String, DataSource> dataSources = buildDataSource(config, iDataSourceBuilderList);

        return dataSources.values().stream().findFirst().orElse(null);
    }


    private void buildConfigGroup(HelpDataSourceConfigGroup config, HelpDataSourceConfig defaultSpringDataSourceConfig) {
        if (defaultSpringDataSourceConfig != null) {
            if (config.getDefaultDataSource() != null) {
                if (config.getMulti() != null && config.getMulti().length > 0) {
                    List<HelpDataSourceConfig> configs = new ArrayList<>(Arrays.asList(config.getMulti()));
                    configs.add(0, defaultSpringDataSourceConfig);
                    config.setMulti(configs.toArray(new HelpDataSourceConfig[0]));
                } else {
                    config.setMulti(new HelpDataSourceConfig[]{defaultSpringDataSourceConfig});
                }
            } else {
                config.setDefaultDataSource(defaultSpringDataSourceConfig);
            }
        }
    }

    private Map<String, DataSource> buildDataSource(HelpDataSourceConfigGroup config, List<IDataSourceBuilder> iDataSourceBuilderList) {
        List<HelpDataSourceConfig> configs = new ArrayList<>();
        if (config.getDefaultDataSource() != null) {
            configs.add(config.getDefaultDataSource());
        }
        if (config.getMulti() != null && config.getMulti().length > 0) {
            configs.addAll(Arrays.asList(config.getMulti()));
        }

        int count = configs.size();

        DefaultDataSourceBuilder defaultDataSourceBuilder = new DefaultDataSourceBuilder();

        Map<String, DataSource> dataSources = new HashMap<>();
        for (int i = 0; i < configs.size(); i++) {
            HelpDataSourceConfig c = configs.get(i);

            String name = null;

            //数据源名称
            if (StringUtil.isNotEmpty(c.getName())) {
                name = c.getName();
            } else {
                name = "dataSource" + i;
            }
            c.setName(name);

            DataSource dataSource = null;
            if (iDataSourceBuilderList != null && iDataSourceBuilderList.size() > 0) {
                for (IDataSourceBuilder builder : iDataSourceBuilderList) {
                    if (builder.support(c)) {
                        dataSource = builder.buildDataSource(c);
                        break;
                    }
                }
            }
            if (dataSource == null) {
                if (defaultDataSourceBuilder.support(c)) {
                    dataSource = defaultDataSourceBuilder.buildDataSource(c);
                }
            }

            if (dataSource == null) {
                throw new UnifyException(UnifyErrorCode.UNKNOW_FAIL, "当前系统不支持数据库[" + c.getUrl() + "]");
            }

            if (count > 1) {
                logger.info("检测到多数据源环境,自动配置[非XA事务型]数据源[" + name + "](" + c.getUrl() + ")");
            } else {
                logger.info("检测到单数据源环境,自动配置数据源(" + c.getUrl() + ")");
            }


            dataSources.put(name, dataSource);
        }
        return dataSources;
    }
}
