package org.springframework.boot.autoconfigure.condition;

import com.help.conditional.ConditionalOnGenericityBean;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.HierarchicalBeanFactory;
import org.springframework.beans.factory.ListableBeanFactory;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.boot.autoconfigure.AutoConfigurationMetadata;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.context.annotation.ConfigurationCondition;
import org.springframework.core.Ordered;
import org.springframework.core.ResolvableType;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.annotation.Order;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.core.type.MethodMetadata;
import org.springframework.util.*;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.*;

/**
 * @author: YuBin-002726
 * @Date: 2019/9/24 16:36
 */
@Order(Ordered.LOWEST_PRECEDENCE - 1)
public class OnGenericityBeanCondition extends FilteringSpringBootCondition implements ConfigurationCondition {

    @Override
    public ConfigurationPhase getConfigurationPhase() {
        return ConfigurationPhase.REGISTER_BEAN;
    }

    @Override
    protected final ConditionOutcome[] getOutcomes(String[] autoConfigurationClasses,
                                                   AutoConfigurationMetadata autoConfigurationMetadata) {
        ConditionOutcome[] outcomes = new ConditionOutcome[autoConfigurationClasses.length];
        for (int i = 0; i < outcomes.length; i++) {
            String autoConfigurationClass = autoConfigurationClasses[i];
            if (autoConfigurationClass != null) {
                Set<String> onBeanTypes = autoConfigurationMetadata.getSet(autoConfigurationClass, "ConditionalOnGenericityBean");
                outcomes[i] = getOutcome(onBeanTypes, ConditionalOnGenericityBean.class);
            }
        }
        return outcomes;
    }

    private ConditionOutcome getOutcome(Set<String> requiredBeanTypes, Class<? extends Annotation> annotation) {
        List<String> missing = filter(requiredBeanTypes, ClassNameFilter.MISSING, getBeanClassLoader());
        if (!missing.isEmpty()) {
            ConditionMessage message = ConditionMessage.forCondition(annotation)
                    .didNotFind("required type", "required types").items(ConditionMessage.Style.QUOTE, missing);
            return ConditionOutcome.noMatch(message);
        }
        return null;
    }

    @Override
    public ConditionOutcome getMatchOutcome(ConditionContext context, AnnotatedTypeMetadata metadata) {
        ConditionMessage matchMessage = ConditionMessage.empty();
        if (metadata.isAnnotated(ConditionalOnGenericityBean.class.getName())) {
            OnGenericityBeanCondition.BeanSearchSpec spec = new OnGenericityBeanCondition.BeanSearchSpec(context, metadata, ConditionalOnGenericityBean.class);
            OnGenericityBeanCondition.MatchResult matchResult = getMatchingBeans(context, spec);
            if (!matchResult.isAllMatched()) {
                String reason = createOnBeanNoMatchReason(matchResult);
                return ConditionOutcome
                        .noMatch(ConditionMessage.forCondition(ConditionalOnGenericityBean.class, spec).because(reason));
            }
            matchMessage = matchMessage.andCondition(ConditionalOnGenericityBean.class, spec).found("bean", "beans")
                    .items(ConditionMessage.Style.QUOTE, matchResult.getNamesOfAllMatches());
        }
        
        return ConditionOutcome.match(matchMessage);
    }

    protected final OnGenericityBeanCondition.MatchResult getMatchingBeans(ConditionContext context, OnGenericityBeanCondition.BeanSearchSpec beans) {
        ConfigurableListableBeanFactory beanFactory = context.getBeanFactory();
        if (beans.getStrategy() == SearchStrategy.ANCESTORS) {
            BeanFactory parent = beanFactory.getParentBeanFactory();
            Assert.isInstanceOf(ConfigurableListableBeanFactory.class, parent,
                    "Unable to use SearchStrategy.ANCESTORS");
            beanFactory = (ConfigurableListableBeanFactory) parent;
        }
        OnGenericityBeanCondition.MatchResult matchResult = new OnGenericityBeanCondition.MatchResult();
        boolean considerHierarchy = beans.getStrategy() != SearchStrategy.CURRENT;
        BeanTypeRegistry.TypeExtractor typeExtractor = beans.getTypeExtractor(context.getClassLoader());
        for (String type : beans.getTypes()) {
            Collection<String> typeMatches = getBeanNamesForType(beanFactory, type, typeExtractor,
                    context.getClassLoader(), considerHierarchy);
            if (typeMatches.isEmpty()) {
                matchResult.recordUnmatchedType(type);
            }
            else {
                matchResult.recordMatchedType(type, typeMatches);
            }
        }

        return matchResult;
    }

    private Collection<String> getBeanNamesForType(ListableBeanFactory beanFactory, String type,
                                                   BeanTypeRegistry.TypeExtractor typeExtractor, ClassLoader classLoader, boolean considerHierarchy) throws LinkageError {
        try {
            return getBeanNamesForType(beanFactory, considerHierarchy, ClassUtils.forName(type, classLoader),
                    typeExtractor);
        }
        catch (ClassNotFoundException | NoClassDefFoundError ex) {
            return Collections.emptySet();
        }
    }

    private Collection<String> getBeanNamesForType(ListableBeanFactory beanFactory, boolean considerHierarchy,
                                                   Class<?> type, BeanTypeRegistry.TypeExtractor typeExtractor) {
        Set<String> result = new LinkedHashSet<>();
        collectBeanNamesForType(result, beanFactory, type, typeExtractor, considerHierarchy);
        return result;
    }

    private void collectBeanNamesForType(Set<String> result, ListableBeanFactory beanFactory, Class<?> type,
                                         BeanTypeRegistry.TypeExtractor typeExtractor, boolean considerHierarchy) {
        BeanTypeRegistry registry = BeanTypeRegistry.get(beanFactory);
        result.addAll(registry.getNamesForType(type, typeExtractor));
        if (considerHierarchy && beanFactory instanceof HierarchicalBeanFactory) {
            BeanFactory parent = ((HierarchicalBeanFactory) beanFactory).getParentBeanFactory();
            if (parent instanceof ListableBeanFactory) {
                collectBeanNamesForType(result, (ListableBeanFactory) parent, type, typeExtractor, considerHierarchy);
            }
        }
    }

    private String createOnBeanNoMatchReason(OnGenericityBeanCondition.MatchResult matchResult) {
        StringBuilder reason = new StringBuilder();
        appendMessageForNoMatches(reason, matchResult.getUnmatchedAnnotations(), "annotated with");
        appendMessageForNoMatches(reason, matchResult.getUnmatchedTypes(), "of type");
        appendMessageForNoMatches(reason, matchResult.getUnmatchedNames(), "named");
        return reason.toString();
    }

    private void appendMessageForNoMatches(StringBuilder reason, Collection<String> unmatched, String description) {
        if (!unmatched.isEmpty()) {
            if (reason.length() > 0) {
                reason.append(" and ");
            }
            reason.append("did not find any beans ");
            reason.append(description);
            reason.append(" ");
            reason.append(StringUtils.collectionToDelimitedString(unmatched, ", "));
        }
    }

    protected static class BeanSearchSpec {

        private final Class<?> annotationType;

        private final List<String> types = new ArrayList<>();

        private final List<String> parameterizedContainers = new ArrayList<>();

        private final SearchStrategy strategy;

        public BeanSearchSpec(ConditionContext context, AnnotatedTypeMetadata metadata, Class<?> annotationType) {
            this(context, metadata, annotationType, null);
        }

        public BeanSearchSpec(ConditionContext context, AnnotatedTypeMetadata metadata, Class<?> annotationType,
                              Class<?> genericContainer) {
            this.annotationType = annotationType;
            MultiValueMap<String, Object> attributes = metadata.getAllAnnotationAttributes(annotationType.getName(),
                    true);
            collect(attributes, "value", this.types);
            collect(attributes, "type", this.types);
            collect(attributes, "parameterizedContainer", this.parameterizedContainers);
            this.strategy = (SearchStrategy) attributes.getFirst("search");
            OnGenericityBeanCondition.BeanTypeDeductionException deductionException = null;
            try {
                if (this.types.isEmpty() ) {
                    addDeducedBeanType(context, metadata, this.types);
                }
            }
            catch (OnGenericityBeanCondition.BeanTypeDeductionException ex) {
                deductionException = ex;
            }
            validate(deductionException);
        }

        protected void validate(OnGenericityBeanCondition.BeanTypeDeductionException ex) {
            if (!hasAtLeastOne(this.types)) {
                String message = getAnnotationName() + " did not specify a bean using type, name or annotation";
                if (ex == null) {
                    throw new IllegalStateException(message);
                }
                throw new IllegalStateException(message + " and the attempt to deduce" + " the bean's type failed", ex);
            }
        }

        private boolean hasAtLeastOne(List<?>... lists) {
            return Arrays.stream(lists).anyMatch((list) -> !list.isEmpty());
        }

        protected final String getAnnotationName() {
            return "@" + ClassUtils.getShortName(this.annotationType);
        }

        protected void collect(MultiValueMap<String, Object> attributes, String key, List<String> destination) {
            List<?> values = attributes.get(key);
            if (values != null) {
                for (Object value : values) {
                    if (value instanceof String[]) {
                        Collections.addAll(destination, (String[]) value);
                    }
                    else {
                        destination.add((String) value);
                    }
                }
            }
        }

        private void addDeducedBeanType(ConditionContext context, AnnotatedTypeMetadata metadata,
                                        final List<String> beanTypes) {
            if (metadata instanceof MethodMetadata && metadata.isAnnotated(Bean.class.getName())) {
                addDeducedBeanTypeForBeanMethod(context, (MethodMetadata) metadata, beanTypes);
            }
        }

        private void addDeducedBeanTypeForBeanMethod(ConditionContext context, MethodMetadata metadata,
                                                     final List<String> beanTypes) {
            try {
                Class<?> returnType = getReturnType(context, metadata);
                beanTypes.add(returnType.getName());
            }
            catch (Throwable ex) {
                throw new OnGenericityBeanCondition.BeanTypeDeductionException(metadata.getDeclaringClassName(), metadata.getMethodName(), ex);
            }
        }

        private Class<?> getReturnType(ConditionContext context, MethodMetadata metadata)
                throws ClassNotFoundException, LinkageError {
            // We should be safe to load at this point since we are in the
            // REGISTER_BEAN phase
            ClassLoader classLoader = context.getClassLoader();
            Class<?> returnType = ClassUtils.forName(metadata.getReturnTypeName(), classLoader);
            if (isParameterizedContainer(returnType, classLoader)) {
                returnType = getReturnTypeGeneric(metadata, classLoader);
            }
            return returnType;
        }

        private Class<?> getReturnTypeGeneric(MethodMetadata metadata, ClassLoader classLoader)
                throws ClassNotFoundException, LinkageError {
            Class<?> declaringClass = ClassUtils.forName(metadata.getDeclaringClassName(), classLoader);
            Method beanMethod = findBeanMethod(declaringClass, metadata.getMethodName());
            return ResolvableType.forMethodReturnType(beanMethod).resolveGeneric();
        }

        private Method findBeanMethod(Class<?> declaringClass, String methodName) {
            Method method = ReflectionUtils.findMethod(declaringClass, methodName);
            if (isBeanMethod(method)) {
                return method;
            }
            return Arrays.stream(ReflectionUtils.getAllDeclaredMethods(declaringClass))
                    .filter((candidate) -> candidate.getName().equals(methodName)).filter(this::isBeanMethod)
                    .findFirst()
                    .orElseThrow(() -> new IllegalStateException("Unable to find bean method " + methodName));
        }

        private boolean isBeanMethod(Method method) {
            return method != null && AnnotatedElementUtils.hasAnnotation(method, Bean.class);
        }

        public BeanTypeRegistry.TypeExtractor getTypeExtractor(ClassLoader classLoader) {
            if (this.parameterizedContainers.isEmpty()) {
                return ResolvableType::resolve;
            }
            return (type) -> {
                Class<?> resolved = type.resolve();
                if (isParameterizedContainer(resolved, classLoader)) {
                    return type.getGeneric().resolve();
                }
                return resolved;
            };
        }

        private boolean isParameterizedContainer(Class<?> type, ClassLoader classLoader) {
            for (String candidate : this.parameterizedContainers) {
                try {
                    if (ClassUtils.forName(candidate, classLoader).isAssignableFrom(type)) {
                        return true;
                    }
                }
                catch (Exception ex) {
                }
            }
            return false;
        }

        public SearchStrategy getStrategy() {
            return (this.strategy != null) ? this.strategy : SearchStrategy.ALL;
        }

        public List<String> getTypes() {
            return this.types;
        }

        @Override
        public String toString() {
            StringBuilder string = new StringBuilder();
            string.append("(");
            if (!this.types.isEmpty()) {
                string.append("types: ");
                string.append(StringUtils.collectionToCommaDelimitedString(this.types));
            }
            string.append("; SearchStrategy: ");
            string.append(this.strategy.toString().toLowerCase(Locale.ENGLISH));
            string.append(")");
            return string.toString();
        }

    }

    private static class SingleCandidateBeanSearchSpec extends OnGenericityBeanCondition.BeanSearchSpec {

        SingleCandidateBeanSearchSpec(ConditionContext context, AnnotatedTypeMetadata metadata,
                                      Class<?> annotationType) {
            super(context, metadata, annotationType);
        }

        @Override
        protected void collect(MultiValueMap<String, Object> attributes, String key, List<String> destination) {
            super.collect(attributes, key, destination);
            destination.removeAll(Arrays.asList("", Object.class.getName()));
        }

        @Override
        protected void validate(OnGenericityBeanCondition.BeanTypeDeductionException ex) {
            Assert.isTrue(getTypes().size() == 1,
                    () -> getAnnotationName() + " annotations must specify only one type (got " + getTypes() + ")");
        }

    }

    protected static final class MatchResult {

        private final Map<String, Collection<String>> matchedAnnotations = new HashMap<>();

        private final List<String> matchedNames = new ArrayList<>();

        private final Map<String, Collection<String>> matchedTypes = new HashMap<>();

        private final List<String> unmatchedAnnotations = new ArrayList<>();

        private final List<String> unmatchedNames = new ArrayList<>();

        private final List<String> unmatchedTypes = new ArrayList<>();

        private final Set<String> namesOfAllMatches = new HashSet<>();

        private void recordMatchedName(String name) {
            this.matchedNames.add(name);
            this.namesOfAllMatches.add(name);
        }

        private void recordUnmatchedName(String name) {
            this.unmatchedNames.add(name);
        }

        private void recordMatchedAnnotation(String annotation, Collection<String> matchingNames) {
            this.matchedAnnotations.put(annotation, matchingNames);
            this.namesOfAllMatches.addAll(matchingNames);
        }

        private void recordUnmatchedAnnotation(String annotation) {
            this.unmatchedAnnotations.add(annotation);
        }

        private void recordMatchedType(String type, Collection<String> matchingNames) {
            this.matchedTypes.put(type, matchingNames);
            this.namesOfAllMatches.addAll(matchingNames);
        }

        private void recordUnmatchedType(String type) {
            this.unmatchedTypes.add(type);
        }

        public boolean isAllMatched() {
            return this.unmatchedAnnotations.isEmpty() && this.unmatchedNames.isEmpty()
                    && this.unmatchedTypes.isEmpty();
        }

        public boolean isAnyMatched() {
            return (!this.matchedAnnotations.isEmpty()) || (!this.matchedNames.isEmpty())
                    || (!this.matchedTypes.isEmpty());
        }

        public Map<String, Collection<String>> getMatchedAnnotations() {
            return this.matchedAnnotations;
        }

        public List<String> getMatchedNames() {
            return this.matchedNames;
        }

        public Map<String, Collection<String>> getMatchedTypes() {
            return this.matchedTypes;
        }

        public List<String> getUnmatchedAnnotations() {
            return this.unmatchedAnnotations;
        }

        public List<String> getUnmatchedNames() {
            return this.unmatchedNames;
        }

        public List<String> getUnmatchedTypes() {
            return this.unmatchedTypes;
        }

        public Set<String> getNamesOfAllMatches() {
            return this.namesOfAllMatches;
        }

    }

    static final class BeanTypeDeductionException extends RuntimeException {

        private BeanTypeDeductionException(String className, String beanMethodName, Throwable cause) {
            super("Failed to deduce bean type for " + className + "." + beanMethodName, cause);
        }

    }

}
