package com.help.storage;

import com.help.common.exception.UnifyErrorCode;
import com.help.common.exception.UnifyException;
import com.help.common.util.FileUtil;
import com.help.common.util.StreamUtils;
import com.help.common.util.StringUtil;

import java.io.*;
import java.util.UUID;

/**
 * HELP平台默认本地文件存储器
 *
 * @author: YuBin-002726
 * @Date: 2019/2/19 13:18
 */
public class LocalFileStorage implements IFileStorage {

    private String root;

    public LocalFileStorage(String root) {
        if (StringUtil.isEmpty(root)) {
            root = "/";
        }
        if (!root.endsWith("/") && !root.endsWith("\\")) {
            root += File.separatorChar;
        }
        this.root = root;
    }

    private String getFileName(String id) {
        String dir = root + id.substring(id.length() - 3);

        FileUtil.mkdir(dir);

        return dir + File.separatorChar + id;
    }

    @Override
    public String save(byte[] file, String originalFileName) {
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(file);
        return save(byteArrayInputStream, originalFileName);
    }

    @Override
    public String save(InputStream is, String originalFileName) {
        String id = UUID.randomUUID().toString().replace("-", "");

        try (FileOutputStream fos = new FileOutputStream(getFileName(id))) {

            StreamUtils.copy(is, fos);

            return id;
        } catch (Exception e) {
            throw new UnifyException(UnifyErrorCode.UNKNOW_FAIL, "文件保存失败:[" + e.getMessage() + "]");
        }
    }

    @Override
    public void delete(String fileId) {
        File file = new File(getFileName(fileId));
        if (file.exists()) {
            file.delete();
        }
    }

    @Override
    public byte[] get(String fileId) {
        try (InputStream is = getInputStream(fileId)) {
            return StreamUtils.copyToByteArray(is);
        } catch (IOException e) {
            throw new UnifyException("文件[" + fileId + "]读取失败");
        }
    }

    @Override
    public InputStream getInputStream(String fileId) {
        try {
            return new FileInputStream(getFileName(fileId));
        } catch (FileNotFoundException e) {
            throw new UnifyException(UnifyErrorCode.NON_DATA, "文件[" + fileId + "]不存在");
        }
    }

    @Override
    public long getSize(String fileId) {
        File file = new File(getFileName(fileId));
        if (file.exists()) {
            return file.length();
        } else {
            throw new UnifyException("文件[" + fileId + "]不存在");
        }
    }

}
