package com.helpframework.controller;

import com.help.annotation.OperationLog;
import com.help.annotation.UnifyAuthorization;
import com.help.common.InvocationResult;
import com.help.common.annotation.ParamValid;
import com.help.common.exception.UnifyErrorCode;
import com.help.common.exception.UnifyException;
import com.help.common.exception.UnifyValidateException;
import com.help.common.util.StringUtil;
import com.help.constant.CommonConstant;
import com.help.constant.OperationType;
import com.help.constant.PermissionConstant;
import com.help.domain.RoleInfo;
import com.help.service.UnionService;
import com.help.storage.editable.IEditableRoleStorage;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping("/role")
@UnifyAuthorization(module = PermissionConstant.CONFIG_ROLE_MENU, moduleName = "角色管理", system = CommonConstant.YES)
public class EditRoleController {

    @Autowired
    private IEditableRoleStorage roleService;
    @Autowired
    private UnionService unionService;

    @UnifyAuthorization(op = PermissionConstant.PERMISSION_ADD)
    @PostMapping(value = "/create.do")
    @OperationLog(value = "创建角色[{0}-{1}]", spel = {"#role.roleNo", "#role.roleName"}, type = OperationType.SYSTEM)
    public InvocationResult create(@ParamValid RoleInfo role) {
        role.setState(CommonConstant.DATA_STATUS_ENABLE);
        roleService.add(role);
        return InvocationResult.SUCCESS;
    }

    @UnifyAuthorization(op = PermissionConstant.PERMISSION_UPDATE)
    @PostMapping(value = "/update.do")
    @OperationLog(value = "修改角色[{0}]", spel = {"#role.roleNo"}, type = OperationType.SYSTEM)
    public InvocationResult update(@ParamValid RoleInfo role) {
        if (StringUtil.isNotEmpty(role.getRoleNo())) {
            role.setState(CommonConstant.DATA_STATUS_ENABLE);

            if (PermissionConstant.ROLE_ADMIN.equalsIgnoreCase(role.getRoleNo()) && !CommonConstant.DATA_STATUS_ENABLE.equalsIgnoreCase(role.getState())) {
                throw new UnifyException(UnifyErrorCode.BUSINESS_FAILD, "不允许禁用管理员[admin]角色");
            } else if (PermissionConstant.ROLE_DEFAULT.equalsIgnoreCase(role.getRoleNo()) && !CommonConstant.DATA_STATUS_ENABLE.equalsIgnoreCase(role.getState())) {
                throw new UnifyException(UnifyErrorCode.BUSINESS_FAILD, "不允许禁用默认角色[default]角色");
            }

            roleService.edit(role);
        } else {
            throw new UnifyValidateException("请求参数不合法,缺少更新条件");
        }

        return InvocationResult.SUCCESS;
    }

    @UnifyAuthorization(op = PermissionConstant.PERMISSION_DELETE)
    @PostMapping(value = "/delete.do")
    @OperationLog(value = "删除角色[{0}]", spel = {"#role.roleNo"}, type = OperationType.SYSTEM)
    public InvocationResult delete(@ParamValid String roleNo) {
        if (PermissionConstant.ROLE_ADMIN.equalsIgnoreCase(roleNo)) {
            throw new UnifyException(UnifyErrorCode.BUSINESS_FAILD, "不允许删除管理员[admin]角色");
        } else if (PermissionConstant.ROLE_DEFAULT.equalsIgnoreCase(roleNo)) {
            throw new UnifyException(UnifyErrorCode.BUSINESS_FAILD, "不允许删除默认角色[default]角色");
        }

        unionService.deleteRole(roleNo);
        return InvocationResult.SUCCESS;
    }

}
