package com.help.web.controller;

import com.help.ITokenContext;
import com.help.LoginInfo;
import com.help.annotation.UnifyAuthorization;
import com.help.common.InvocationResult;
import com.help.common.UnifyPageData;
import com.help.common.UnifyPageInfo;
import com.help.common.annotation.ParamValid;
import com.help.common.exception.UnifyErrorCode;
import com.help.constant.CommonConstant;
import com.help.constant.PermissionConstant;
import com.help.constant.RoleAffiliationConstant;
import com.help.domain.UserInfo;
import com.help.storage.IUserAffiliationStorage;
import com.help.storage.IUserStorage;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author: YuBin-002726
 * @Date: 2019/1/30 9:01
 */
@UnifyAuthorization(module = PermissionConstant.CONFIG_USER_MENU, moduleName = "用户管理", system = CommonConstant.YES)
@RestController
@RequestMapping("/user")
public class UserController {

    @Autowired
    private IUserStorage iUserStorage;
    @Autowired
    private IUserAffiliationStorage iUserAffiliationStorage;
    @Autowired
    ITokenContext<LoginInfo> iTokenContext;

    @UnifyAuthorization(op = PermissionConstant.PERMISSION_DETAIL)
    @PostMapping(value = "/detail.do")
    public InvocationResult detail(@ParamValid String userNo) {
        UserInfo userInfo = iUserStorage.get(userNo);
        if (userInfo != null) {
            return new InvocationResult(userInfo);
        } else {
            return new InvocationResult(UnifyErrorCode.NON_DATA, "用户[" + userNo + "]不存在");
        }
    }

    @UnifyAuthorization(op = PermissionConstant.PERMISSION_VISIT)
    @PostMapping(value = "/listByPage.do")
    public InvocationResult listByPage(UnifyPageInfo page, @RequestParam(required = false) String org, @RequestParam(required = false) String dept, @RequestParam(required = false) String user) {
        UnifyPageData<UserInfo> list = iUserStorage.searchByPage(user, org, dept, page);
        return new InvocationResult(list);
    }


    @PostMapping(value = "/listByUser.do")
    public InvocationResult<List<Map>> listByUser(String userNo) {
        List<Map> list = iUserAffiliationStorage.listRoleAffiliationByUser(userNo).stream().map(p -> {
            Map<String, String> map = new HashMap<>();
            map.put("userNo", p.getUserNo());
            map.put("roleNo", p.getRoleNo());
            if (RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType())) {
                map.put("orgNo", p.getAffiNo());
            } else if (RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType())) {
                map.put("deptNo", p.getAffiNo());
            }
            return map;
        }).collect(Collectors.toList());

        return new InvocationResult<>(list);
    }

    @PostMapping(value = "/listPageByUser.do")
    public InvocationResult<UnifyPageData<Map>> listPageByUser(String userNo, UnifyPageInfo pageInfo) {
        List<Map> list = iUserAffiliationStorage.listRoleAffiliationByUser(userNo).stream().map(p -> {
            Map<String, String> map = new HashMap<>();
            map.put("userNo", p.getUserNo());
            map.put("roleNo", p.getRoleNo());
            if (RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType())) {
                map.put("orgNo", p.getAffiNo());
            } else if (RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType())) {
                map.put("deptNo", p.getAffiNo());
            }
            return map;
        }).collect(Collectors.toList());

        UnifyPageData<Map> data = new UnifyPageData<>();
        data.setList(list.stream().skip((pageInfo.getPageIndex() - 1) * pageInfo.getPageSize()).limit(pageInfo.getPageSize()).collect(Collectors.toList()));
        data.setPageIndex(pageInfo.getPageIndex());
        data.setPageSize(pageInfo.getPageSize());
        data.setTotalSize(list.size());
        return new InvocationResult<>(data);
    }

    @PostMapping(value = "/listByRole.do")
    public InvocationResult<List<Map>> listByRole(String roleNo) {
        List<Map> list = iUserAffiliationStorage.listRoleAffiliationByRole(roleNo).stream().map(p -> {
            Map<String, String> map = new HashMap<>();
            map.put("userNo", p.getUserNo());
            map.put("roleNo", p.getRoleNo());
            if (RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType())) {
                map.put("orgNo", p.getAffiNo());
            } else if (RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType())) {
                map.put("deptNo", p.getAffiNo());
            }
            return map;
        }).collect(Collectors.toList());

        return new InvocationResult<>(list);
    }

    @PostMapping(value = "/listPageByRole.do")
    public InvocationResult<UnifyPageData<Map>> listPageByRole(String roleNo, UnifyPageInfo pageInfo) {
        List<Map> list = iUserAffiliationStorage.listRoleAffiliationByRole(roleNo).stream().map(p -> {
            Map<String, String> map = new HashMap<>();
            map.put("userNo", p.getUserNo());
            map.put("roleNo", p.getRoleNo());
            if (RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType())) {
                map.put("orgNo", p.getAffiNo());
            } else if (RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType())) {
                map.put("deptNo", p.getAffiNo());
            }
            return map;
        }).collect(Collectors.toList());

        UnifyPageData<Map> data = new UnifyPageData<>();
        data.setList(list.stream().skip((pageInfo.getPageIndex() - 1) * pageInfo.getPageSize()).limit(pageInfo.getPageSize()).collect(Collectors.toList()));
        data.setPageIndex(pageInfo.getPageIndex());
        data.setPageSize(pageInfo.getPageSize());
        data.setTotalSize(list.size());
        return new InvocationResult<>(data);
    }

    @PostMapping(value = "/listByDept.do")
    public InvocationResult<List<Map>> listByDept(String deptNo) {
        List<Map> list = iUserAffiliationStorage.listRoleAffiliationByDept(deptNo).stream().map(p -> {
            Map<String, String> map = new HashMap<>();
            map.put("userNo", p.getUserNo());
            map.put("roleNo", p.getRoleNo());
            if (RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType())) {
                map.put("orgNo", p.getAffiNo());
            } else if (RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType())) {
                map.put("deptNo", p.getAffiNo());
            }
            return map;
        }).collect(Collectors.toList());

        return new InvocationResult<>(list);
    }

    @PostMapping(value = "/listPageByDept.do")
    public InvocationResult<UnifyPageData<Map>> listPageByDept(String deptNo, UnifyPageInfo pageInfo) {
        List<Map> list = iUserAffiliationStorage.listRoleAffiliationByDept(deptNo).stream().map(p -> {
            Map<String, String> map = new HashMap<>();
            map.put("userNo", p.getUserNo());
            map.put("roleNo", p.getRoleNo());
            if (RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType())) {
                map.put("orgNo", p.getAffiNo());
            } else if (RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType())) {
                map.put("deptNo", p.getAffiNo());
            }
            return map;
        }).collect(Collectors.toList());

        UnifyPageData<Map> data = new UnifyPageData<>();
        data.setList(list.stream().skip((pageInfo.getPageIndex() - 1) * pageInfo.getPageSize()).limit(pageInfo.getPageSize()).collect(Collectors.toList()));
        data.setPageIndex(pageInfo.getPageIndex());
        data.setPageSize(pageInfo.getPageSize());
        data.setTotalSize(list.size());
        return new InvocationResult<>(data);
    }

    @PostMapping(value = "/listByOrg.do")
    public InvocationResult<List<Map>> listByOrg(String orgNo) {
        List<Map> list = iUserAffiliationStorage.listRoleAffiliationByOrg(orgNo).stream().map(p -> {
            Map<String, String> map = new HashMap<>();
            map.put("userNo", p.getUserNo());
            map.put("roleNo", p.getRoleNo());
            if (RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType())) {
                map.put("orgNo", p.getAffiNo());
            } else if (RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType())) {
                map.put("deptNo", p.getAffiNo());
            }
            return map;
        }).collect(Collectors.toList());

        return new InvocationResult<>(list);
    }

    @PostMapping(value = "/listPageByOrg.do")
    public InvocationResult<UnifyPageData<Map>> listPageByOrg(String orgNo, UnifyPageInfo pageInfo) {
        List<Map> list = iUserAffiliationStorage.listRoleAffiliationByOrg(orgNo).stream().map(p -> {
            Map<String, String> map = new HashMap<>();
            map.put("userNo", p.getUserNo());
            map.put("roleNo", p.getRoleNo());
            if (RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType())) {
                map.put("orgNo", p.getAffiNo());
            } else if (RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType())) {
                map.put("deptNo", p.getAffiNo());
            }
            return map;
        }).collect(Collectors.toList());

        UnifyPageData<Map> data = new UnifyPageData<>();
        data.setList(list.stream().skip((pageInfo.getPageIndex() - 1) * pageInfo.getPageSize()).limit(pageInfo.getPageSize()).collect(Collectors.toList()));
        data.setPageIndex(pageInfo.getPageIndex());
        data.setPageSize(pageInfo.getPageSize());
        data.setTotalSize(list.size());
        return new InvocationResult<>(data);
    }

}
