package com.help.web.controller;

import com.help.ITokenContext;
import com.help.LoginInfo;
import com.help.annotation.OperationLog;
import com.help.common.HelpManageUtil;
import com.help.common.ICacheable;
import com.help.common.IParamReader;
import com.help.common.InvocationResult;
import com.help.common.annotation.ParamValid;
import com.help.common.exception.UnifyErrorCode;
import com.help.common.exception.UnifyException;
import com.help.common.tuple.Tuple2;
import com.help.common.util.BeanConvert;
import com.help.common.util.StringUtil;
import com.help.annotation.UnifyAuthorization;
import com.help.config.HelpManageConfig;
import com.help.constant.CacheConstant;
import com.help.constant.CommonConstant;
import com.help.constant.OperationType;
import com.help.constant.UserConfigConstant;
import com.help.constraint.IHelpOAuthClient;
import com.help.domain.*;
import com.help.plugins.DeptDicExtension;
import com.help.plugins.OrgTreeDicExtension;
import com.help.service.*;
import com.help.storage.*;
import com.help.storage.editable.IEditableUserStorage;
import com.help.web.module.DicItemFormated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.PostConstruct;
import javax.imageio.ImageIO;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletResponse;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 系统控制器
 *
 * @author YuBin-002726
 */
@RestController
public class SystemController {
    private static final Logger logger = LoggerFactory.getLogger(SystemController.class);

    private final String VALID_CODE_COOKIE_NAME = "valid-code-key";

    @Autowired(required = false)
    IHelpOAuthClient ioAuthClient;
    @Autowired
    ITokenContext<LoginInfo> iTokenContext;
    @Autowired
    IUserStorage iUserStorage;
    @Autowired
    IRoleStorage iRoleStorage;
    @Autowired
    OrgTreeDicExtension orgTreeDicExtension;
    @Autowired
    DeptDicExtension deptDicExtension;
    @Autowired
    ValidCodeService validCodeService;
    @Autowired
    private UserConfigService pUserConfigService;
    @Autowired
    private PermissionService permissionService;
    @Autowired(required = false)
    private CacheManager cacheManager;
    @Autowired
    private DictionaryService dictionaryService;
    @Autowired
    private LoginService loginService;
    @Autowired(required = false)
    private IParamReader iParamReader;

    @Autowired
    private HelpManageConfig helpManageConfig;

    @PostConstruct
    public void init() {
        if (cacheManager == null) {
            logger.warn("未找到缓存管理器,系统缓存可能未启用");
        }
    }

    @PostMapping("/sysConfig.do")
    public InvocationResult sysConfig() {
        Map<String, String> config = new HashMap<>();
        config.put("passwordEditable", (iUserStorage instanceof IEditableUserStorage) ? "1" : "0");
        config.put("asyncOrgTree", helpManageConfig.isAsyncOrgTree() ? "1" : "0");
        config.put("strongCipher", helpManageConfig.isStrongCipher() ? "1" : "0");
        config.put("loginValidCode", helpManageConfig.isLoginValidCode() ? "1" : "0");

        return new InvocationResult(config);
    }

    @GetMapping(value = "/validCode", produces = MediaType.IMAGE_JPEG_VALUE)
    public ResponseEntity<ByteArrayResource> validCode(HttpServletResponse response, @CookieValue(value = VALID_CODE_COOKIE_NAME, required = false) String validKey, @RequestParam(value = "rnd", required = false) String rnd) {
        //移除原有验证码
        validCodeService.removeKey(validKey);

        Tuple2<String, BufferedImage> validCode = HelpManageUtil.getValidCodeImage(6);

        byte[] data = null;
        try (ByteArrayOutputStream bio = new ByteArrayOutputStream()) {
            ImageIO.write(validCode.getT2(), "JPG", bio);
            data = bio.toByteArray();
        } catch (IOException e) {
            throw new UnifyException(UnifyErrorCode.UNKNOW_FAIL, "验证码生成失败");
        }

        String key = null;
        if (StringUtil.isNotEmpty(rnd)) {
            key = validCodeService.createKey(validCode.getT1(), rnd);
        } else {
            key = validCodeService.createKey(validCode.getT1());
        }
        Cookie cookie = new Cookie(VALID_CODE_COOKIE_NAME, key);
        cookie.setPath("/");
        cookie.setHttpOnly(true);
        response.addCookie(cookie);
        ResponseEntity<ByteArrayResource> responseEntity = new ResponseEntity<>(new ByteArrayResource(data), HttpStatus.OK);
        return responseEntity;
    }

    //异步登陆处理器
    @PostMapping(value = "/loginAsync.do")
    @OperationLog(value = "用户登录", userSpel = "#username", type = OperationType.BUSINESS)
    public InvocationResult loginAsync(@ParamValid String username, @ParamValid String password, String validCode, @CookieValue(value = VALID_CODE_COOKIE_NAME, required = false) String validKey, @RequestParam(value = "rnd", required = false) String rnd) {
        if (password.length() < 6 || password.length() > 16) {
            return new InvocationResult(UnifyErrorCode.VALIDATE_FAILD, "密码长度必须是6-16位");
        }

        //校验验证码
        if (helpManageConfig.isLoginValidCode()) {
            if (StringUtil.isEmpty(validKey)) {
                validKey = rnd;
            }
            if (StringUtil.isEmpty(validCode)) {
                return new InvocationResult(UnifyErrorCode.VALIDATE_FAILD, "验证码不能为空");
            } else if (StringUtil.isEmpty(validKey)) {
                return new InvocationResult(UnifyErrorCode.VALIDATE_FAILD, "验证码随机Key不能为空");
            } else if (!validCodeService.valid(validKey, validCode)) {
                return new InvocationResult(UnifyErrorCode.VALIDATE_FAILD, "验证码错误");
            }
        }

        LoginInfo old = iTokenContext.getCurrentToken();
        LoginInfo loginInfo = loginService.getToken(username, password);

        //检测强密码
        if (helpManageConfig.isStrongCipher() && !HelpManageUtil.strongCipherValid(password)) {
            //插入弱密码标记
            pUserConfigService.save(username, UserConfigConstant.WEAK_PASSWORD_TAG, "1");
        }

        if (old != null && old.getUserId().equalsIgnoreCase(loginInfo.getUserId())) {
            return new InvocationResult(old.getToken());
        } else {
            //注销现有
            if (old != null) {
                iTokenContext.removeToken();
            }

            iTokenContext.saveToken(loginInfo);

            return new InvocationResult(loginInfo.getToken());
        }
    }

    //登出操作
    @PostMapping(value = "/logout.do")
    public InvocationResult logout() {
        iTokenContext.removeToken();
        return InvocationResult.SUCCESS;
    }

    //列出当前登陆用户相关信息
    @UnifyAuthorization
    @PostMapping(value = "/userInfo.do")
    public InvocationResult userInfo() {
        LoginInfo loginInfo = iTokenContext.getCurrentToken();

        Map<String, Object> info = BeanConvert.toMap(loginInfo);

        //列出操作权限
        info.put("roles", permissionService.listRoles(loginInfo.getUserId()));

        //取用户机构
        info.put("orgs", permissionService.listOrgs(loginInfo.getUserId()));

        //取用户部门
        info.put("depts", permissionService.listDepts(loginInfo.getUserId()));

        //角色归属
        info.put("affi", permissionService.listAffiliations(loginInfo.getUserId()));

        //操作权限
        info.put("ops", permissionService.listModuleOps(loginInfo));

        info.put("passwordEditable", (iUserStorage instanceof IEditableUserStorage) ? "1" : "0");

        return new InvocationResult(info);
    }


    //切换登陆机构/部门
    @UnifyAuthorization
    @PostMapping(value = "/switchLoginOrg.do")
    public InvocationResult switchLoginOrg(String orgNo, String deptNo) {
        LoginInfo loginInfo = iTokenContext.getCurrentToken();
        if (StringUtil.isNotEmpty(orgNo)) {
            List<OrgInfoBase> userOrgs = permissionService.listOrgs(loginInfo.getUserId());
            OrgInfoBase loginOrg = userOrgs.stream().filter(p -> p.getOrgNo().equalsIgnoreCase(orgNo)).findFirst().orElse(null);
            if (loginOrg != null) {
                loginInfo.setLoginOrg(orgNo);
                pUserConfigService.save(loginInfo.getUserId(), UserConfigConstant.LAST_LOGIN_ORG, orgNo);
            } else {
                return new InvocationResult(UnifyErrorCode.BUSINESS_FAILD.getCode(), "机构切换失败，当前用户不属于该机构");
            }
        } else {
            loginInfo.setLoginOrg(null);
            pUserConfigService.delete(loginInfo.getUserId(), UserConfigConstant.LAST_LOGIN_ORG);
        }

        if (StringUtil.isNotEmpty(deptNo)) {
            List<DeptInfo> userDept = permissionService.listDepts(loginInfo.getUserId());
            DeptInfo loginDept = userDept.stream().filter(p -> p.getDeptNo().equalsIgnoreCase(deptNo)).findFirst().orElse(null);
            if (loginDept != null) {
                loginInfo.setLoginDept(deptNo);
                pUserConfigService.save(loginInfo.getUserId(), UserConfigConstant.LAST_LOGIN_DEPT, deptNo);
            } else {
                return new InvocationResult(UnifyErrorCode.BUSINESS_FAILD.getCode(), "机构切换失败，当前用户不属于该机构");
            }
        } else {
            loginInfo.setLoginDept(null);
            pUserConfigService.delete(loginInfo.getUserId(), UserConfigConstant.LAST_LOGIN_DEPT);
        }

        //取可用角色
        List<String> roles = permissionService.listRoles(loginInfo.getUserId(), loginInfo.getLoginOrg(), loginInfo.getLoginDept(), false).stream().map(
                p -> p.getRoleNo()
        ).collect(Collectors.toList());
        loginInfo.setLoginRoles(roles);
        iTokenContext.saveToken(loginInfo);

        return InvocationResult.SUCCESS;
    }

    private List<Map<String, Object>> fillChild(List<TreeDicItem> all) {

        List<Map<String, Object>> list = all.stream().map(p -> {
            Map<String, Object> item = new HashMap<>();
            item.put("code", p.getCode());
            item.put("text", p.getText());
            item.put("child", new ArrayList<>());
            item.put("parent", p.getParent());
            return item;
        }).collect(Collectors.toList());

        Iterator<Map<String, Object>> iterator = list.iterator();
        while (iterator.hasNext()) {
            Map<String, Object> item = iterator.next();
            //寻找父节点
            Map<String, Object> parent = list.stream().filter(p -> p.get("code").toString().equals(item.get("parent"))).findAny().orElse(null);
            if (parent != null) {
                ((List) parent.get("child")).add(item);
            } else {
                item.remove("parent");
            }
        }

        return list.stream().filter(p -> StringUtil.isEmpty(p.get("parent"))).collect(Collectors.toList());
    }

    //获取格式化后的字典数据
    @PostMapping(value = "/listDicFormat.do")
    public InvocationResult listDicFormat(@ParamValid String[] dicType) {
        Map<String, List<?>> dics = new HashMap<>();
        for (String t : dicType) {
            List<DicItem> list = dictionaryService.list(t);
            if (list != null && list.size() > 0) {
                dics.put(t, list.stream().map(p -> new DicItemFormated(p)).collect(Collectors.toList()));
            } else {
                dics.put(t, fillChild(dictionaryService.listTree(t)));
            }
        }
        return new InvocationResult(dics);
    }

    //获取字典数据
    @UnifyAuthorization
    @PostMapping(value = "/listDic.do")
    public InvocationResult listDic(@ParamValid String[] dicType) {
        Map<String, List<?>> dics = new HashMap<>();
        for (String t : dicType) {
            List<DicItem> list = dictionaryService.list(t);
            if (list != null && list.size() > 0) {
                dics.put(t, list);
            } else {
                dics.put(t, dictionaryService.listTree(t));
            }
        }
        return new InvocationResult(dics);
    }

    //获取树形字典数据
    @UnifyAuthorization
    @PostMapping(value = "/listTreeDic.do")
    public InvocationResult listTreeDic(@ParamValid String[] dicType) {
        Map<String, List<TreeDicItem>> dics = new HashMap<>();
        for (String t : dicType) {
            dics.put(t, dictionaryService.listTree(t));
        }
        return new InvocationResult(dics);
    }

    @UnifyAuthorization
    @PostMapping(value = "/listDeptDicFormatByLegal.do")
    public InvocationResult listDeptDicFormatByLegal(@RequestParam(value = "legal", required = false) String legal) {
        List<DicItemFormated> list = deptDicExtension.listByLegal(legal, true).stream().map(p -> new DicItemFormated(p)).collect(Collectors.toList());
        return new InvocationResult(list);
    }

    @UnifyAuthorization
    @PostMapping(value = "/listOrgDicFormatByLegal.do")
    public InvocationResult listOrgDicFormatByLegal(@RequestParam(value = "legal", required = false) String legal) {
        List<TreeDicItem> list = orgTreeDicExtension.listByLegal(legal, true);
        return new InvocationResult(fillChild(list));
    }

    //清空缓存池
    @UnifyAuthorization(module = "config_cache", moduleName = "缓存清理", op = "clean", opName = "清空缓存", system = CommonConstant.YES)
    @PostMapping(value = "/cleanCache.do")
    @OperationLog(value = "清空缓存", type = OperationType.SYSTEM)
    public InvocationResult cleanCache(String name) {
        if (cacheManager != null) {
            if (StringUtil.isEmpty(name)) {
                if (iParamReader != null && iParamReader instanceof ICacheable) {
                    ((ICacheable) iParamReader).refresh();
                }
                CacheConstant[] names = CacheConstant.values();
                for (CacheConstant c : names) {
                    //清除所有缓存数据
                    cacheManager.getCache(c.getName()).clear();
                }
                return InvocationResult.SUCCESS;
            } else {
                if (!"session".equals(name)) {
                    Cache cache = cacheManager.getCache(name);
                    if (cache != null) {
                        cache.clear();
                        return InvocationResult.SUCCESS;
                    } else {
                        throw new UnifyException(UnifyErrorCode.VALIDATE_FAILD.getCode(), "缓存不存在");
                    }
                } else {
                    return new InvocationResult(UnifyErrorCode.NON_INTERFACE, "当前系统配置不支持此操作");
                }
            }
        }

        return new InvocationResult(UnifyErrorCode.BUSINESS_FAILD, "系统缓存未启用");
    }
}
