package com.help.web.controller;

import com.help.ModuleAnalasys;
import com.help.annotation.OperationLog;
import com.help.annotation.UnifyAuthorization;
import com.help.common.InvocationResult;
import com.help.common.annotation.ParamValid;
import com.help.common.util.StringUtil;
import com.help.constant.CommonConstant;
import com.help.constant.OperationType;
import com.help.constant.PermissionConstant;
import com.help.service.PermissionService;
import com.help.web.module.Module;
import com.help.web.module.Operation;
import com.help.web.module.RoleModuleOp;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 权限配置控制器
 *
 * @author: YuBin-002726
 * @Date: 2019/3/5 17:37
 */
@RestController
@RequestMapping("/permission")
@UnifyAuthorization(module = "config_permission", moduleName = "权限管理", system = CommonConstant.YES)
public class PermissionController {

    @Autowired
    ModuleAnalasys moduleAnalasys;

    @Autowired
    PermissionService permissionService;

    //列出所有功能模块
    @UnifyAuthorization(op = PermissionConstant.PERMISSION_VISIT)
    @PostMapping(value = "/listAll.do")
    public InvocationResult<List<Module>> listAll() {
        return new InvocationResult(moduleAnalasys.list());
    }

    //列出模块下所有操作
    @UnifyAuthorization(op = PermissionConstant.PERMISSION_VISIT)
    @PostMapping(value = "/listByModule.do")
    public InvocationResult<List<Operation>> listByModule(String moduleNo) {
        Module module = moduleAnalasys.get(moduleNo);
        if (module != null) {
            return new InvocationResult<>(module.getOperations());
        } else {
            return new InvocationResult<>(new ArrayList<>());
        }
    }

    /**
     * 列出某个角色在某个模块下的所有功能权限
     *
     * @param moduleNo
     * @param roleNo
     * @return
     */
    @UnifyAuthorization(op = PermissionConstant.PERMISSION_VISIT)
    @PostMapping(value = "/list.do")
    public InvocationResult list(@ParamValid String moduleNo, @ParamValid String roleNo) {
        return new InvocationResult(permissionService.listPermission(roleNo, moduleNo));
    }

    //为某角色授予某菜单的操作权限
    @UnifyAuthorization(op = "grant", opName = "授权")
    @PostMapping(value = "/grant.do")
    @OperationLog(value = "为角色[{0}]授权[{1}-{2}]", spel = {"#permission.roleNo", "#permission.moduleNo", "#permission.op"}, type = OperationType.SYSTEM)
    public InvocationResult grant(@ParamValid RoleModuleOp permission) {
        permissionService.grantPermission(permission);
        return InvocationResult.SUCCESS;
    }

    //撤回某角色的某菜单的操作权限
    @UnifyAuthorization(op = "grant", opName = "授权")
    @PostMapping(value = "/revoke.do")
    @OperationLog(value = "撤回角色[{0}]权限[{1}-{2}]", spel = {"#permission.roleNo", "#permission.moduleNo", "#permission.op"}, type = OperationType.SYSTEM)
    public InvocationResult revoke(@ParamValid RoleModuleOp permission) {
        permissionService.revokePermission(permission);
        return InvocationResult.SUCCESS;
    }

    //为某角色授予某菜单的操作权限
    @UnifyAuthorization(op = "grant", opName = "授权")
    @PostMapping(value = "/save.do")
    @OperationLog(value = "为角色[{0}]批量授权[{1}]", spel = {"#roleNo", "#moduleNo"}, type = OperationType.SYSTEM)
    public InvocationResult save(@ParamValid String roleNo, @ParamValid String moduleNo, String[] ops) {
        if (ops == null) {
            ops = new String[0];
        }
        List<String> list = new ArrayList<>();
        for (String o : ops) {
            if (StringUtil.isNotEmpty(o)) {
                list.addAll(Arrays.asList(o.split(",")));
            }
        }
        permissionService.reset(roleNo, moduleNo, list.toArray(new String[0]));

        return InvocationResult.SUCCESS;
    }

    //将某个角色的所有操作权限复制给另一个角色
    @UnifyAuthorization(op = "grant", opName = "授权")
    @PostMapping(value = "/copy.do")
    @OperationLog(value = "复制角色[{0}]权限到[{1}]", spel = {"#roleFrom", "#roleTo"}, type = OperationType.SYSTEM)
    public InvocationResult copy(@ParamValid String roleFrom, @ParamValid String roleTo) {
        permissionService.copy(roleFrom, roleTo);
        return InvocationResult.SUCCESS;
    }

}
