package com.help.web.controller;

import com.help.ITokenContext;
import com.help.LoginInfo;
import com.help.common.InvocationResult;
import com.help.common.exception.UnifyException;
import com.help.domain.FileInfo;
import com.help.service.FileService;
import com.help.common.exception.UnifyErrorCode;
import com.help.common.util.StringUtil;
import com.help.annotation.UnifyAuthorization;
import com.help.TranscodeBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.InputStreamResource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * 文件服务控制器
 */
@Controller
@UnifyAuthorization
@RequestMapping("/file")
public class FileController {

    @Autowired
    private FileService fileService;
    @Autowired
    ITokenContext<LoginInfo> iTokenContext;
    @Autowired
    TranscodeBuilder transcodeBuilder;

    @ResponseBody
    @PostMapping(value = "/upload.do")
    public InvocationResult<List<String>> upload(@RequestParam("file") MultipartFile[] files) {
        String inputer = iTokenContext.getCurrentToken().getUserId();

        List<String> ids = new ArrayList<>();

        if (files != null && files.length > 0) {
            for (MultipartFile f : files) {
                String id = fileService.saveFile(f, inputer);
                if (StringUtil.isNotEmpty(id)) {
                    ids.add(id);
                }
            }
        }

        return new InvocationResult(ids);
    }

    @ResponseBody
    @PostMapping(value = "/info.do")
    public InvocationResult<FileInfo> info(String fileId) {
        return new InvocationResult<>(fileService.getInfo(fileId));
    }

    @ResponseBody
    @PostMapping(value = "/infos.do")
    public InvocationResult<Collection<FileInfo>> infos(String[] fileId) {
        Collection<FileInfo> list = fileService.getInfos(fileId).values();
        return new InvocationResult<>(list);
    }

    @ResponseBody
    @PostMapping(value = "/delete.do")
    public InvocationResult delete(String fileId) {
        LoginInfo currentUser = iTokenContext.getCurrentToken();
        FileInfo pFile = fileService.getInfo(fileId);
        if (currentUser.getUserId().equalsIgnoreCase(pFile.getInputUser())) {
            fileService.deleteFile(fileId);
            return InvocationResult.SUCCESS;
        } else {
            return new InvocationResult(UnifyErrorCode.BUSINESS_FAILD, "只有文件上传者能删除此文件");
        }
    }

    @UnifyAuthorization(valid = false)
    @GetMapping(value = "/download/{fileId}")
    public ResponseEntity<?> download(@PathVariable("fileId") String fileid, @RequestParam(value = "token", required = false) String token) {

        try {
            LoginInfo currentUser = iTokenContext.getCurrentToken();
            if (currentUser == null) {
                if (StringUtil.isNotEmpty(token)) {
                    currentUser = iTokenContext.getToken(token);
                } else {
                    return new ResponseEntity(new InvocationResult(UnifyErrorCode.RQEUIRED_LOGIN, "下载文件失败,您当前尚未登录"), HttpStatus.NOT_FOUND);
                }
            }

            if (currentUser != null) {

                FileInfo pFile = fileService.getInfo(fileid);

                if (pFile != null) {
                    HttpHeaders headers = new HttpHeaders();
                    String nameEncoded = URLEncoder.encode(pFile.getOriginalName(), "UTF-8");
                    headers.set("Content-Disposition", "attachment; filename=\"" + nameEncoded + "\"; filename*=UTF-8''" + nameEncoded);
                    headers.set("Content-Type", pFile.getFileMime());

                    return new ResponseEntity<>(new InputStreamResource(fileService.getFileStream(fileid)), headers, HttpStatus.OK);
                } else {
                    return new ResponseEntity(new InvocationResult(UnifyErrorCode.NON_DATA, "下载文件失败,文件不存在"), HttpStatus.NOT_FOUND);
                }
            } else {
                return new ResponseEntity(new InvocationResult(UnifyErrorCode.RQEUIRED_LOGIN, "下载文件失败,您当前尚未登录"), HttpStatus.NOT_FOUND);
            }
        } catch (UnifyException e) {
            return new ResponseEntity(new InvocationResult(e), HttpStatus.NOT_FOUND);
        } catch (Exception e) {
            return new ResponseEntity(new InvocationResult(UnifyErrorCode.NON_DATA, "下载文件失败"), HttpStatus.NOT_FOUND);
        }
    }

}
