package com.help.web.aop;

import com.help.ITokenContext;
import com.help.LoginInfo;
import com.help.annotation.OperationLog;
import com.help.common.InvocationResult;
import com.help.common.exception.UnifyErrorCode;
import com.help.common.exception.UnifyException;
import com.help.common.util.StringUtil;
import com.help.config.HelpManageConfig;
import com.help.constant.OperationLogStorage;
import com.help.domain.POperationLog;
import com.help.service.POperationLogService;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.LocalVariableTableParameterNameDiscoverer;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.Expression;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.util.Date;

/**
 * HELP平台操作日志自动记录器
 *
 * @author: YuBin-002726
 * @Date: 2019/9/26 16:02
 */
@Aspect
public class HelpOperationLogAspect implements Ordered {

    private Logger logger = LoggerFactory.getLogger(GlobalAuthorizationCheckAspect.class);

    private ExpressionParser parser = new SpelExpressionParser();

    private LocalVariableTableParameterNameDiscoverer discoverer = new LocalVariableTableParameterNameDiscoverer();

    @Autowired(required = false)
    ITokenContext<LoginInfo> iTokenContext;

    @Autowired
    HelpManageConfig helpManageConfig;

    @Autowired
    HttpServletRequest request;

    @Autowired
    POperationLogService pOperationLogService;

    @PostConstruct
    public void init() {
        OperationLogStorage[] storage = helpManageConfig.getOperationLogStorage();
        if (storage != null && storage.length > 0) {
            for (OperationLogStorage s : storage) {
                if (s == OperationLogStorage.DATABASE) {
                    db = true;
                } else if (s == OperationLogStorage.LOGFILE) {
                    logfile = true;
                }
            }
        }
    }

    private boolean logfile;
    private boolean db;

    //切入Controller/RestController下带有@OperationLog注解的方法
    @Around(" @within(org.springframework.stereotype.Controller) && @annotation(com.help.annotation.OperationLog)" +
            "|| @within(org.springframework.web.bind.annotation.RestController) &&  @annotation(com.help.annotation.OperationLog)")
    public Object execute(ProceedingJoinPoint pjp) throws Throwable {
        if (db || logfile) {
            //是否明确为执行失败
            Boolean succ = null;

            //是否需要记录日志
            boolean requireRecord = true;

            try {
                Object obj = pjp.proceed();
                if (obj != null && obj instanceof InvocationResult && !UnifyErrorCode.SUCCESS.getCode().equalsIgnoreCase(((InvocationResult) obj).getState())) {
                    succ = false;
                }
                return obj;
            } catch (Exception e) {
                succ = false;
                if (e instanceof UnifyException && ((UnifyException) e).getErrorCode() == UnifyErrorCode.RQEUIRED_LOGIN) {
                    //对于未授权用户的访问无需写入记录
                    requireRecord = false;
                }
                throw e;
            } finally {
                if (requireRecord) {
                    Method method = ((MethodSignature) pjp.getSignature()).getMethod();

                    OperationLog operationLog = AnnotationUtils.findAnnotation(method, OperationLog.class);

                    String str = operationLog.value();

                    String[] params = null;

                    if (operationLog.spel() != null && operationLog.spel().length > 0 || StringUtil.isNotEmpty(operationLog.userSpel())) {
                        params = discoverer.getParameterNames(method);
                        for (int i = 0; i < operationLog.spel().length; i++) {
                            String result = parseSpel(pjp.getArgs(), params, operationLog.spel()[i], "");
                            if (result == null) {
                                result = "";
                            }
                            str = str.replaceAll("\\{" + i + "\\}", result);
                        }
                    }

                    String userId = null;
                    if (StringUtil.isNotEmpty(operationLog.userSpel())) {
                        userId = parseSpel(pjp.getArgs(), params, operationLog.userSpel(), "");
                    } else {
                        if (iTokenContext != null) {
                            try {
                                userId = iTokenContext.getCurrentToken().getUserId();
                            } catch (Exception e) {

                            }
                        }
                    }
                    if (StringUtil.isEmpty(userId)) {
                        userId = "Unknow";
                    }

                    if (succ == null && operationLog.succWhenUnThrow()) {
                        succ = true;
                    }
                    String ip = getIpAddr(request);
                    if (logfile) {
                        logger.info("用户[{}({})]  执行 [{}]操作 {} {}", userId, ip, operationLog.type().getName(), str, succ == null ? "结果未知" : (succ ? "成功" : "失败"));
                    }
                    if (db) {
                        POperationLog log = new POperationLog();
                        log.setContent(str);
                        log.setGmtCreate(new Date());
                        log.setIpAddr(ip);
                        log.setOpType(operationLog.type().getCode());
                        log.setSuccess(succ == null ? "" : (succ ? "1" : "0"));
                        log.setUserNo(userId);
                        pOperationLogService.add(log);
                    }
                }
            }
        } else {
            return pjp.proceed();
        }

    }

    private String parseSpel(Object[] arguments, String[] argumentNames, String spel, String defaultResult) {

        EvaluationContext context = new StandardEvaluationContext();
        for (int len = 0; len < argumentNames.length; len++) {
            context.setVariable(argumentNames[len], arguments[len]);
        }
        try {
            Expression expression = parser.parseExpression(spel);
            return expression.getValue(context, String.class);
        } catch (Exception e) {
            return defaultResult;
        }
    }

    private String getIpAddr(HttpServletRequest request) {
        String ip = request.getHeader("x-forwarded-for");
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
            if (ip.equals("127.0.0.1")) {
                //根据网卡取本机配置的IP
                InetAddress inet = null;
                try {
                    inet = InetAddress.getLocalHost();
                } catch (Exception e) {
                    e.printStackTrace();
                }
                ip = inet.getHostAddress();
            }
        }
        // 多个代理的情况，第一个IP为客户端真实IP,多个IP按照','分割
        if (ip != null && ip.length() > 15) {
            if (ip.indexOf(",") > 0) {
                ip = ip.substring(0, ip.indexOf(","));
            }
        }
        return ip;
    }


    @Override
    public int getOrder() {
        return 15;
    }

}
