package com.help.web.aop;

import com.help.ITokenContext;
import com.help.LoginInfo;
import com.help.common.exception.UnifyErrorCode;
import com.help.common.exception.UnifyException;
import com.help.common.util.StringUtil;
import com.help.constant.CommonConstant;
import com.help.constant.PermissionConstant;
import com.help.annotation.UnifyAuthorization;
import com.help.service.PermissionService;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.Ordered;

import java.lang.reflect.Method;
import java.util.List;

/**
 * 全局控制器调用权限验证器
 *
 * @author YuBin-002726
 */
@Aspect
public class GlobalAuthorizationCheckAspect implements Ordered {

    private Logger logger = LoggerFactory.getLogger(GlobalAuthorizationCheckAspect.class);

    @Autowired
    ITokenContext<LoginInfo> iTokenContext;

    @Autowired
    private PermissionService permissionService;

    //切入具有UnifyAuthorization的Controller/RestController或具有UnifyAuthorization的方法
    @Before("@within(com.help.annotation.UnifyAuthorization) && @within(org.springframework.stereotype.Controller)" +
            "|| @within(com.help.annotation.UnifyAuthorization) && @within(org.springframework.web.bind.annotation.RestController)" +
            "|| @annotation(com.help.annotation.UnifyAuthorization)")
    public void execute(JoinPoint pjp) {
        // 代理对象
        String target = pjp.getSignature().getDeclaringTypeName();
        Method method = ((MethodSignature) pjp.getSignature()).getMethod();
        //获取代理方法上的注解
        UnifyAuthorization anno = method.getAnnotation(UnifyAuthorization.class);
        //获取代理类上的注解
        UnifyAuthorization classAnno = (UnifyAuthorization) pjp.getSignature().getDeclaringType().getAnnotation(UnifyAuthorization.class);
        //若方法上无某注解属性则获取类上的注解

        if (anno != null && anno.valid() || anno == null && classAnno != null && classAnno.valid()) {
            String module = null;
            String op = null;
            String system = null;
            String moduleName = null;
            String opName = null;
            boolean grantToAll = false;

            if (anno != null) {
                module = anno.module();
                moduleName = anno.moduleName();
                op = anno.op();
                opName = anno.opName();
                system = anno.system();
                grantToAll = anno.grantToAll();
            }
            if (classAnno != null) {
                if (StringUtil.isEmpty(module)) {
                    module = classAnno.module();
                    moduleName = classAnno.moduleName();
                }
                if (StringUtil.isEmpty(op)) {
                    op = classAnno.op();
                    opName = classAnno.opName();
                }
                if (StringUtil.isEmpty(system)) {
                    system = classAnno.system();
                }
                if (anno == null) {
                    grantToAll = classAnno.grantToAll();
                }
            }

            logger.debug("正在验证[" + target + "]的[" + method.getName() + "]调用权限,所属模块:[" + StringUtil.nvl(moduleName, "") + module + "],功能:[" + StringUtil.nvl(opName, "") + op + "]");
            //进行登陆验证
            LoginInfo current = iTokenContext.getCurrentToken();
            if (current == null) {
                throw new UnifyException(UnifyErrorCode.RQEUIRED_LOGIN);
            }

            //对于授权给所有用户的权限,无需验证
            if (grantToAll) {
                logger.debug("所有用户默认拥有该权限,跳过验证");
            } else if (current.getLoginRoles() != null && current.getLoginRoles().stream().anyMatch(p -> p.equalsIgnoreCase(PermissionConstant.ROLE_ADMIN)) && CommonConstant.YES.equalsIgnoreCase(system)) {
                //对于系统模块,且角色为系统管理员,直接验证通过
                logger.debug("系统管理员默认拥有该权限,跳过验证");
            } else if (StringUtil.isNotEmptyAll(module, op)) {
                //进行权限验证
                if (!isAuthorization(current, module, op)) {
                    throw new UnifyException(UnifyErrorCode.UN_AUTHORIZATION);
                }
            }
        }
    }

    //验证操作权限
    private boolean isAuthorization(LoginInfo loginInfo, String module, String op) {
        List<String> rs = loginInfo.getLoginRoles();
        if (rs == null || rs.size() == 0) {
            return false;
        } else {
            return permissionService.listModuleOps(loginInfo).stream().anyMatch(p -> p.getModule().equalsIgnoreCase(module) && p.getOp().equalsIgnoreCase(op));
        }
    }

    @Override
    public int getOrder() {
        return 100;
    }
}
