package com.help.service;

import com.help.LoginInfo;
import com.help.ModuleAnalasys;
import com.help.common.ICacheable;
import com.help.common.util.StringUtil;
import com.help.constant.CommonConstant;
import com.help.constant.RoleAffiliationConstant;
import com.help.dao.batch.BatchHelper;
import com.help.dao.*;
import com.help.domain.OrgInfoBase;
import com.help.domain.RoleInfo;
import com.help.domain.*;
import com.help.constant.CacheConstant;
import com.help.constant.PermissionConstant;
import com.help.storage.*;
import com.help.web.module.Module;
import com.help.web.module.Operation;
import com.help.web.module.RoleModuleOp;
import org.springframework.aop.framework.AopContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.CacheManager;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 用户/权限业务类
 * 本类主要实现用户与权限的查询与角色操作权限的授予与撤回等操作
 *
 * @author YuBin-002726
 */
@Service
public class PermissionService implements ICacheable {

    @Autowired
    private PMenuMapper pMenuMapper;

    @Autowired
    private PRoleModuleOpMapper PRoleModuleOpMapper;
    @Autowired
    private IUserAffiliationStorage iUserAffiliationStorage;
    @Autowired
    private IRoleStorage iRoleStorage;
    @Autowired
    private IOrgStorage iOrgStorage;
    @Autowired
    private IDeptStorage iDeptStorage;
    @Autowired
    private ModuleAnalasys moduleAnalasys;

    @Autowired
    CacheManager cacheManager;

    @Autowired
    BatchHelper batchHelper;

    /**
     * 列出用户拥有的所有角色(无论当前登录状态是否可用)
     *
     * @param userNo
     * @return
     */
    public List<RoleInfo> listRoles(String userNo) {
        return this.listRoles(userNo, null, null, true);
    }

    /**
     * 列出用户在指定机构和部门下的可用角色(包括无归属角色和指定机构/部门下的角色)
     *
     * @param userNo       用户编号
     * @param orgNo        机构编号
     * @param deptNo       部门编号
     * @param emptyWithAll 若机构/部门为空,是否查询所有机构/部门,若此参数为false,则当传递空值时,将过滤掉机构/部门的角色数据
     * @return
     */
    public List<RoleInfo> listRoles(String userNo, String orgNo, String deptNo, boolean emptyWithAll) {
        Stream<UserAffiliationInfo> stream = ((PermissionService) AopContext.currentProxy()).listAffiliations(userNo).stream();
        if (StringUtil.isNotEmpty(orgNo) && StringUtil.isNotEmpty(deptNo)) {
            //机构部门都非空,则直接取公共角色并机构角色并部门角色
            stream = stream.filter(p ->
                    RoleAffiliationConstant.ROLE_AFFILIATION_NONE.equalsIgnoreCase(p.getAffiType())
                            || RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType()) && orgNo.equalsIgnoreCase(p.getAffiNo())
                            || RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType()) && deptNo.equalsIgnoreCase(p.getAffiNo())
            );
        } else if (StringUtil.isNotEmpty(orgNo)) {
            //机构非空而部门为空.根据emptyWithAll决定查询所有部门还是移除部门角色
            if (emptyWithAll) {
                stream = stream.filter(p ->
                        RoleAffiliationConstant.ROLE_AFFILIATION_NONE.equalsIgnoreCase(p.getAffiType())
                                || RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType()) && orgNo.equalsIgnoreCase(p.getAffiNo())
                                || RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType())
                );
            } else {
                stream = stream.filter(p ->
                        RoleAffiliationConstant.ROLE_AFFILIATION_NONE.equalsIgnoreCase(p.getAffiType())
                                || RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType()) && orgNo.equalsIgnoreCase(p.getAffiNo())
                );
            }
        } else if (StringUtil.isNotEmpty(deptNo)) {
            //机构为空而部门非.根据emptyWithAll决定查询所有机构还是移除机构角色
            if (emptyWithAll) {
                stream = stream.filter(p ->
                        RoleAffiliationConstant.ROLE_AFFILIATION_NONE.equalsIgnoreCase(p.getAffiType())
                                || RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType()) && deptNo.equalsIgnoreCase(p.getAffiNo())
                                || RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType())
                );
            } else {
                stream = stream.filter(p ->
                        RoleAffiliationConstant.ROLE_AFFILIATION_NONE.equalsIgnoreCase(p.getAffiType())
                                || RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType()) && deptNo.equalsIgnoreCase(p.getAffiNo())
                );
            }
        } else {
            //两者都为空
            if (emptyWithAll) {
                stream = stream.filter(p ->
                        RoleAffiliationConstant.ROLE_AFFILIATION_NONE.equalsIgnoreCase(p.getAffiType())
                                || RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType())
                                || RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType())
                );
            } else {
                stream = stream.filter(p ->
                        RoleAffiliationConstant.ROLE_AFFILIATION_NONE.equalsIgnoreCase(p.getAffiType())
                );
            }
        }

        List<String> roles = stream.map(p -> p.getRoleNo()).distinct().collect(Collectors.toList());
        if (!roles.contains("default")) {
            roles.add("default");
        }
        return iRoleStorage.getAll(roles.toArray(new String[0]));
    }


    /**
     * 列出用户详细角色归属
     *
     * @param userNo
     * @return
     */
    public List<UserAffiliationInfo> listAffiliations(String userNo) {
        return iUserAffiliationStorage.listRoleAffiliationByUser(userNo);
    }

    /**
     * 列出用户可操作机构
     *
     * @param userNo
     * @return
     */
    public List<OrgInfoBase> listOrgs(String userNo) {
        List<String> orgs = this.listAffiliations(userNo).stream().filter(p ->
                RoleAffiliationConstant.ROLE_AFFILIATION_ORG.equalsIgnoreCase(p.getAffiType())
        ).map(p -> p.getAffiNo()).distinct().collect(Collectors.toList());

        return iOrgStorage.getAll(orgs.toArray(new String[0]));
    }

    /**
     * 列出用户可操作部门
     *
     * @param userNo
     * @return
     */
    public List<DeptInfo> listDepts(String userNo) {
        List<String> depts = this.listAffiliations(userNo).stream().filter(p ->
                RoleAffiliationConstant.ROLE_AFFILIATION_DEPT.equalsIgnoreCase(p.getAffiType())
        ).map(p -> p.getAffiNo()).distinct().collect(Collectors.toList());

        return iDeptStorage.getAll(depts.toArray(new String[0]));
    }

    /**
     * 列出所有可用菜单
     *
     * @return
     */
    @Deprecated
    public List<PMenu> listAllMenus() {
        PMenuExample example = new PMenuExample();
        example.setOrderByClause("ORDERNO ASC");

        return pMenuMapper.selectByExample(example);
    }

//    /**
//     * 列出用户可见的菜单
//     *
//     * @param userNo 用户编号
//     * @param orgNo  机构编号,若空则查询所有机构
//     * @param deptNo 部门编号,若空则查询所有部门
//     * @return
//     */
//    @Deprecated
//    public List<PMenu> listMenus(String userNo, String orgNo, String deptNo) {
//        //列出用户具有的所有权限
//        List<Operation> operations = this.listModuleOps(userNo, orgNo, deptNo);
//        //查询所有菜单
//        List<PMenu> allMenus = this.listAllMenus();
//
//        return allMenus.stream().filter(p ->
//                StringUtil.isEmpty(p.getModuleNo()) || operations.stream().anyMatch(o -> o.getModule().equalsIgnoreCase(p.getModuleNo()) && o.getOp().equalsIgnoreCase(StringUtil.nvl(p.getOp(), PermissionConstant.PERMISSION_VISIT)))
//        ).collect(Collectors.toList());
//    }

    /**
     * 列出当前登录用户可用的所有菜单
     *
     * @param loginInfo
     * @return
     * @deprecated 该方法已弃用
     */
    @Deprecated
    public List<PMenu> listMenus(LoginInfo loginInfo) {
        //列出用户具有的所有权限
        List<Operation> operations = this.listModuleOps(loginInfo);
        //查询所有菜单
        List<PMenu> allMenus = this.listAllMenus();

        return allMenus.stream().filter(p ->
                StringUtil.isEmpty(p.getModuleNo()) || operations.stream().anyMatch(o -> o.getModule().equalsIgnoreCase(p.getModuleNo()) && o.getOp().equalsIgnoreCase(StringUtil.nvl(p.getOp(), PermissionConstant.PERMISSION_VISIT)))
        ).collect(Collectors.toList());
    }

    /**
     * 列出用户当前可用的模块操作权限
     *
     * @param loginInfo 用户登录信息
     * @return
     */
    public List<Operation> listModuleOps(LoginInfo loginInfo) {
        List<String> rs = loginInfo.getLoginRoles();
        if (rs == null || rs.size() == 0) {
            return new ArrayList<>();
        } else {
            return ((PermissionService) AopContext.currentProxy()).listModuleOpsByRolesCache(rs.stream().sorted().collect(Collectors.toList()).toArray(new String[0]));
        }
    }

//    /**
//     * 列出用户具有的模块操作权限
//     *
//     * @param userNo 用户编号
//     * @param orgNo  机构编号,若空则查询所有机构
//     * @param deptNo 部门编号,若空则查询所有部门
//     * @return
//     */
//    public List<Operation> listModuleOps(String userNo, String orgNo, String deptNo) {
//        //列出用户角色
//        List<String> roles = this.listRoles(userNo, orgNo, deptNo, false).stream().map(p -> p.getRoleNo()).collect(Collectors.toList());
//        if (roles != null && roles.size() > 0) {
//            return ((PermissionService) AopContext.currentProxy()).listModuleOpsByRolesCache(roles.toArray(new String[0]));
//        } else {
//            return new ArrayList<>();
//        }
//    }

    /**
     * 列出角色具有的所有功能模块权限(缓存)
     *
     * @param roleNos
     * @return
     */
    @Cacheable(value = CacheConstant.CACHE_PERMISSION)
    public List<Operation> listModuleOpsByRolesCache(String... roleNos) {
        if (roleNos != null && roleNos.length > 0) {
            List<String> roles = Arrays.asList(roleNos);

            //查数据库中的功能数据 并排除不存在的功能
            List<Operation> allOps = moduleAnalasys.listOps();

            PRoleModuleOpExample PRoleModuleOpExample = new PRoleModuleOpExample();
            PRoleModuleOpExample.createCriteria().andRoleNoIn(roles);

            List<Operation> list = PRoleModuleOpMapper.selectByExample(PRoleModuleOpExample).stream()
                    .filter(p -> allOps.stream().anyMatch(q -> p.getModuleNo().equalsIgnoreCase(q.getModule()) && p.getOpNo().equalsIgnoreCase(q.getOp())))
                    .map(p -> {
                        Operation info = new Operation();
                        info.setModule(p.getModuleNo());
                        info.setOp(p.getOpNo());
                        return info;
                    }).collect(Collectors.toList());

            //如果角色是admin,则合并系统功能
            if (roles.stream().anyMatch(p -> PermissionConstant.ROLE_ADMIN.equalsIgnoreCase(p))) {
                return sum(list, moduleAnalasys.listSystemOps(), moduleAnalasys.listGrantToAll());
            } else {
                return sum(list, moduleAnalasys.listGrantToAll());
            }
        } else {
            return moduleAnalasys.listGrantToAll();
        }
    }

    /**
     * 列出某角色在模块下的功能权限
     *
     * @param roleNo
     * @param moduleNo
     * @return
     */
    public List<Operation> listPermission(String roleNo, String moduleNo) {
        if (StringUtil.isEmpty(moduleNo)) {
            return new ArrayList<>();
        }

        //查数据库中的功能数据 并排除不存在的功能
        List<Operation> allOps = moduleAnalasys.listOps();

        PRoleModuleOpExample PRoleModuleOpExample = new PRoleModuleOpExample();
        PRoleModuleOpExample.createCriteria().andRoleNoEqualTo(roleNo).andModuleNoEqualTo(moduleNo);
        List<Operation> list = PRoleModuleOpMapper.selectByExample(PRoleModuleOpExample).stream()
                .filter(p -> allOps.stream().anyMatch(q -> p.getModuleNo().equalsIgnoreCase(q.getModule()) && p.getOpNo().equalsIgnoreCase(q.getOp())))
                .map(p -> {
                    Operation info = new Operation();
                    info.setModule(p.getModuleNo());
                    info.setOp(p.getOpNo());
                    return info;
                }).collect(Collectors.toList());


        //如果角色是admin,则合并系统功能
        if (PermissionConstant.ROLE_ADMIN.equalsIgnoreCase(roleNo)) {
            Module module = moduleAnalasys.get(moduleNo);
            if (module != null) {
                if (CommonConstant.YES.equalsIgnoreCase(module.getSystem())) {
                    return sum(list, module.getOperations());
                }
            }
        }
        return sum(list, moduleAnalasys.listGrantToAll().stream().filter(p -> p.getModule().equalsIgnoreCase(moduleNo)).collect(Collectors.toList()));
    }

    /**
     * 为某个角色授予模块功能权限
     *
     * @param op
     */
    public void grantPermission(RoleModuleOp op) {
        PRoleModuleOp PRoleModuleOp = new PRoleModuleOp();
        PRoleModuleOp.setRoleNo(op.getRoleNo());
        PRoleModuleOp.setOpNo(op.getOp());
        PRoleModuleOp.setModuleNo(op.getModuleNo());
        PRoleModuleOpMapper.insert(PRoleModuleOp);
    }

    /**
     * 撤回某个角色的菜单操作权限
     *
     * @param op
     */
    public void revokePermission(RoleModuleOp op) {
        PRoleModuleOpExample example = new PRoleModuleOpExample();
        example.createCriteria().andRoleNoEqualTo(op.getRoleNo()).andModuleNoEqualTo(op.getModuleNo()).andOpNoEqualTo(op.getOp());
        PRoleModuleOpMapper.deleteByExample(example);
    }

    /**
     * 将某个角色的菜单操作权限复制给另一个角色
     *
     * @param roleFrom 复制的权限源
     * @param roleTo   要复制到的角色
     */
    @Transactional
    public void copy(String roleFrom, final String roleTo) {
        PRoleModuleOpExample example = new PRoleModuleOpExample();
        example.createCriteria().andRoleNoEqualTo(roleFrom);
        List<PRoleModuleOp> ops = PRoleModuleOpMapper.selectByExample(example);

        example.clear();
        example.createCriteria().andRoleNoEqualTo(roleTo);
        List<PRoleModuleOp> opOld = PRoleModuleOpMapper.selectByExample(example);

        Iterator<PRoleModuleOp> its = ops.iterator();
        while (its.hasNext()) {
            PRoleModuleOp o = its.next();
            for (PRoleModuleOp old : opOld) {
                if (old.getOpNo().equals(o.getOpNo()) && old.getModuleNo().equals(o.getModuleNo())) {
                    its.remove();
                    break;
                }
            }
        }

        batchHelper.getBatchExecuter(PRoleModuleOpMapper.class).execute(ops, (dao, entity) -> {
            entity.setRoleNo(roleTo);
            dao.insert(entity);
        });
    }

    /**
     * 将某个角色的菜单操作权限重设
     *
     * @param roleNo
     * @param menuno
     * @param actno
     */
    @Transactional
    public void reset(String roleNo, String menuno, String[] actno) {
        List<PRoleModuleOp> list = new ArrayList<>();
        for (String a : actno) {
            PRoleModuleOp op = new PRoleModuleOp();
            op.setOpNo(a);
            op.setModuleNo(menuno);
            op.setRoleNo(roleNo);
            list.add(op);
        }

        PRoleModuleOpExample exp = new PRoleModuleOpExample();
        exp.createCriteria().andRoleNoEqualTo(roleNo).andModuleNoEqualTo(menuno);
        PRoleModuleOpMapper.deleteByExample(exp);

        batchHelper.getBatchExecuter(PRoleModuleOpMapper.class).execute(list, (dao, entity) -> dao.insert(entity));
    }

    @Override
    public void refresh() {
        cacheManager.getCache(CacheConstant.CACHE_PERMISSION).clear();
    }


    //合并
    private List<Operation> sum(List<Operation>... list) {
        List<Operation> result = new ArrayList<>();

        for (List<Operation> ops : list) {
            for (Operation op : ops) {
                if (!result.stream().anyMatch(p -> p.getModule().equalsIgnoreCase(op.getModule()) && p.getOp().equalsIgnoreCase(op.getOp()))) {
                    Operation o = new Operation();
                    o.setModule(op.getModule());
                    o.setOp(op.getOp());

                    result.add(o);
                }
            }
        }

        return result;
    }
}
