package com.help.service;

import com.help.constant.PermissionConstant;
import com.help.constraint.ISearchable;
import com.help.common.util.StringUtil;
import com.help.dao.PMenuMapper;
import com.help.dao.PRoleModuleOpMapper;
import com.help.dao.batch.BatchHelper;
import com.help.dao.batch.IExecuteSingleAction;
import com.help.domain.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * 菜单数据操作类
 * 本业务类主要用于操作菜单基础数据
 *
 * @author YuBin-002726
 */
@Service
public class MenuService implements ISearchable<PMenu, PMenuExample> {

    @Autowired
    private PMenuMapper pMenuMapper;
    @Autowired
    private PRoleModuleOpMapper pRoleMenuOpMapper;
    @Autowired
    private BatchHelper batchHelper;

    public PMenu getMenu(String menuno) {
        PMenu menu = pMenuMapper.selectByPrimaryKey(menuno);
        if (menu != null && StringUtil.isNotEmpty(menu.getModuleNo()) && StringUtil.isEmpty(menu.getOp())) {
            menu.setOp(PermissionConstant.PERMISSION_VISIT);
        }
        return menu;
    }

    /**
     * 列出所有菜单
     *
     * @return
     */
    public List<PMenu> list() {
        PMenuExample example = new PMenuExample();
        example.setOrderByClause("ORDERNO ASC");
        List<PMenu> list = pMenuMapper.selectByExample(example);
        for (PMenu menu : list) {
            if (menu != null && StringUtil.isNotEmpty(menu.getModuleNo()) && StringUtil.isEmpty(menu.getOp())) {
                menu.setOp(PermissionConstant.PERMISSION_VISIT);
            }
        }
        return list;
    }

    public List<PMenu> list(PMenuExample example, String... columns) {
        List<PMenu> list = null;
        if (columns == null || columns.length == 0) {
            list = pMenuMapper.selectByExample(example);
        } else {
            list = pMenuMapper.selectColumnsByExample(example, columns);
        }
        for (PMenu menu : list) {
            if (menu != null && StringUtil.isNotEmpty(menu.getModuleNo()) && StringUtil.isEmpty(menu.getOp())) {
                menu.setOp(PermissionConstant.PERMISSION_VISIT);
            }
        }
        return list;
    }

    /**
     * 列出某个菜单下所有直接子菜单
     *
     * @param superMenuNo
     * @return
     */
    public List<PMenu> list(String superMenuNo) {
        PMenuExample pMenuExample = new PMenuExample();
        pMenuExample.createCriteria().andSuperMenunoEqualTo(superMenuNo);
        pMenuExample.setOrderByClause("ORDERNO ASC");
        List<PMenu> list = pMenuMapper.selectByExample(pMenuExample);
        for (PMenu menu : list) {
            if (menu != null && StringUtil.isNotEmpty(menu.getModuleNo()) && StringUtil.isEmpty(menu.getOp())) {
                menu.setOp(PermissionConstant.PERMISSION_VISIT);
            }
        }
        return list;
    }

    /**
     * 添加一个菜单 同时自动添加默认菜单操作(增,删,改,查,访问)
     *
     * @param pMenu
     */
    public void add(final PMenu pMenu) {
        if (pMenu.getOrderno() == null) {
            pMenu.setOrderno(0);
        }
        if (StringUtil.isEmpty(pMenu.getSuperMenuno())) {
            pMenu.setSuperMenuno(null);
        }
        if (StringUtil.isEmpty(pMenu.getIcon())) {
            pMenu.setIcon(null);
        }
        if (StringUtil.isEmpty(pMenu.getUrl())) {
            pMenu.setUrl(null);
        }

        if (StringUtil.isNotEmpty(pMenu.getModuleNo()) && StringUtil.isEmpty(pMenu.getOp())) {
            pMenu.setOp(PermissionConstant.PERMISSION_VISIT);
        }
        pMenuMapper.insert(pMenu);
    }

    /**
     * 删除一个菜单 并递归删除其下所有子菜单
     *
     * @param menuno 菜单编号
     */
    @Transactional
    public void delete(String menuno) {
        //获取该菜单下所有子菜单并删除
        List<PMenu> childs = list(menuno);
        for (PMenu m : childs) {
            delete(m.getMenuno());
        }

        //删除与该菜单有关的权限
        PRoleModuleOpExample pRoleMenuOpExample = new PRoleModuleOpExample();
        pRoleMenuOpExample.createCriteria().andModuleNoEqualTo(menuno);
        pRoleMenuOpMapper.deleteByExample(pRoleMenuOpExample);

        //删除菜单本身
        pMenuMapper.deleteByPrimaryKey(menuno);
    }

    /**
     * 将某个菜单及其子菜单移动到另一个菜单下作为其子菜单
     *
     * @param menuno
     * @param parentMenuNo
     */
    public void appentTo(String menuno, String parentMenuNo) {
        PMenu pMenu = new PMenu();
        pMenu.setMenuno(menuno);
        pMenu.setSuperMenuno(parentMenuNo);
        //获取最大的序号
        PMenuExample example = new PMenuExample();
        example.createCriteria().andSuperMenunoEqualTo(parentMenuNo).andMenunoNotEqualTo(menuno);
        Integer maxOrder = pMenuMapper.maxOrderByExample(example);
        if (maxOrder != null) {
            pMenu.setOrderno(maxOrder.intValue() + 1);
        } else {
            pMenu.setOrderno(1);
        }

        pMenuMapper.updateColumnsByPrimaryKey(pMenu, "menuno", "superMenuno", "orderno");
    }

    /**
     * 将某个菜单移动到指定菜单前面
     *
     * @param menuno
     * @param target
     */
    @Transactional
    public void moveBefore(String menuno, String target) {
        PMenu current = new PMenu();
        current.setMenuno(menuno);

        //获取关联节点
        PMenu targetMenu = pMenuMapper.selectByPrimaryKey(target);
        //获取父节点ID
        String parentNode = StringUtil.nvl(targetMenu.getSuperMenuno());
        current.setSuperMenuno(parentNode);
        //获取所有子节点
        PMenuExample example = new PMenuExample();
        if (parentNode != null) {
            example.createCriteria().andSuperMenunoEqualTo(parentNode).andMenunoNotEqualTo(menuno);
        } else {
            example.createCriteria().andSuperMenunoIsEmpty().andMenunoNotEqualTo(menuno);
        }
        example.setOrderByClause("ORDERNO ASC");
        final List<PMenu> childs = pMenuMapper.selectByExample(example);
        //重新为其排序
        int currentOrder = 0;
        for (PMenu m : childs) {
            if (m.getMenuno().equals(targetMenu.getMenuno())) {
                current.setOrderno(currentOrder);
                currentOrder++;
            }
            m.setOrderno(currentOrder);
            currentOrder++;
        }

        //更新当前菜单
        pMenuMapper.updateColumnsByPrimaryKey(current, "superMenuno", "orderno");

        //更新其他菜单的排序
        batchHelper.getBatchExecuter(PMenuMapper.class).execute(childs, new IExecuteSingleAction<PMenuMapper, PMenu>() {

            @Override
            public void execute(PMenuMapper dao, PMenu entity) {
                dao.updateColumnsByPrimaryKey(entity, "orderno");
            }
        });
    }

    /**
     * 将某个菜单移动到指定菜单后面
     *
     * @param menuno
     * @param target
     */
    @Transactional
    public void moveAfter(String menuno, String target) {
        PMenu current = new PMenu();
        current.setMenuno(menuno);

        //获取关联节点
        PMenu targetMenu = pMenuMapper.selectByPrimaryKey(target);
        //获取父节点ID
        String parentNode = StringUtil.nvl(targetMenu.getSuperMenuno());
        current.setSuperMenuno(parentNode);
        //获取所有子节点
        PMenuExample example = new PMenuExample();
        if (parentNode != null) {
            example.createCriteria().andSuperMenunoEqualTo(parentNode).andMenunoNotEqualTo(menuno);
        } else {
            example.createCriteria().andSuperMenunoIsEmpty().andMenunoNotEqualTo(menuno);
        }
        example.setOrderByClause("ORDERNO ASC");
        final List<PMenu> childs = pMenuMapper.selectByExample(example);
        //重新为其排序
        int currentOrder = 0;
        for (PMenu m : childs) {
            m.setOrderno(currentOrder);
            if (m.getMenuno().equals(targetMenu.getMenuno())) {
                currentOrder++;
                current.setOrderno(currentOrder);
            }
            currentOrder++;
        }

        //更新当前菜单
        pMenuMapper.updateColumnsByPrimaryKey(current, "superMenuno", "orderno");

        //更新其他菜单的排序
        batchHelper.getBatchExecuter(PMenuMapper.class).execute(childs, new IExecuteSingleAction<PMenuMapper, PMenu>() {

            @Override
            public void execute(PMenuMapper dao, PMenu entity) {
                dao.updateColumnsByPrimaryKey(entity, "orderno");
            }
        });
    }

    /**
     * 通过菜单编号修改菜单所有信息
     *
     * @param pMenu
     */
    public void updateMenu(PMenu pMenu) {
        if (pMenu.getOrderno() == null) {
            pMenu.setOrderno(0);
        }
        if (StringUtil.isEmpty(pMenu.getSuperMenuno())) {
            pMenu.setSuperMenuno(null);
        }
        if (StringUtil.isEmpty(pMenu.getIcon())) {
            pMenu.setIcon(null);
        }
        if (StringUtil.isEmpty(pMenu.getUrl())) {
            pMenu.setUrl(null);
        }
        if (StringUtil.isNotEmpty(pMenu.getModuleNo()) && StringUtil.isEmpty(pMenu.getOp())) {
            pMenu.setOp(PermissionConstant.PERMISSION_VISIT);
        }
        pMenuMapper.updateByPrimaryKey(pMenu);
    }

}
