package com.help.service;

import com.help.common.exception.UnifyException;
import com.help.common.util.MimeUtil;
import com.help.common.util.StringUtil;
import com.help.dao.PFileMapper;
import com.help.domain.FileInfo;
import com.help.domain.FileInfoBase;
import com.help.domain.PFile;
import com.help.domain.PFileExample;
import com.help.storage.IFileStorage;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

/**
 * HELP管理系统附件服务
 *
 * @author: YuBin-002726
 * @Date: 2019/2/19 13:26
 */
@Service
public class FileService implements IFileService {

    @Autowired
    PFileMapper pFileMapper;
    @Autowired
    IFileStorage iFileStorage;

    /**
     * 保存文件,自动解析文件MIME
     *
     * @param f
     * @param inputUser
     * @return
     */
    public String saveFile(MultipartFile f, String inputUser) {
        String fileName = f.getOriginalFilename();
        if (StringUtil.isNotEmpty(fileName)) {
            FileInfoBase file = new FileInfoBase();
            file.setOriginalName(fileName);
            file.setInputUser(inputUser);
            file.setFileMime(f.getContentType());

            try {
                return saveFile(file, f.getInputStream());
            } catch (UnifyException e) {
                throw e;
            } catch (Exception e) {
                throw new UnifyException("文件上传失败:[" + e.getMessage() + "]");
            }
        } else {
            return null;
        }
    }

    /**
     * 保存文件,若MIME不存在,则自动解析文件MIME
     *
     * @param base
     * @param data
     * @return
     */
    public String saveFile(FileInfoBase base, byte[] data) {
        if (StringUtil.isEmpty(base.getFileMime())) {
            base.setFileMime(MimeUtil.getMimetype(base.getOriginalName()));
        }
        return saveFile(base, new ByteArrayInputStream(data));
    }

    /**
     * 保存文件,若MIME不存在,则自动解析文件MIME
     * 本方法不会关闭inputStream流
     *
     * @param base
     * @param inputStream
     * @return
     */
    public String saveFile(FileInfoBase base, InputStream inputStream) {
        String id = iFileStorage.save(inputStream, base.getOriginalName());

        PFile file = new PFile();
        file.setInputUser(base.getInputUser());
        file.setOriginalName(base.getOriginalName());
        file.setFileMime(base.getFileMime());
        if (StringUtil.isEmpty(base.getFileMime())) {
            file.setFileMime(MimeUtil.getMimetype(base.getOriginalName()));
        }
        file.setGmtCreate(new Date());
        file.setFileSize(iFileStorage.getSize(id));
        file.setFileId(id);

        pFileMapper.insert(file);

        return id;
    }

    /**
     * 删除文件
     *
     * @param fileId
     */
    public void deleteFile(String fileId) {
        pFileMapper.deleteByPrimaryKey(fileId);

        iFileStorage.delete(fileId);
    }

    /**
     * 获取文件信息
     *
     * @param fileId
     * @return
     */
    public FileInfo getInfo(String fileId) {
        return transform(pFileMapper.selectByPrimaryKey(fileId));
    }

    /**
     * 获取一组文件信息
     *
     * @param fileIds
     * @return
     */
    public Map<String, FileInfo> getInfos(String... fileIds) {
        if (fileIds == null || fileIds.length == 0) {
            return new HashMap<>();
        } else {
            PFileExample example = new PFileExample();
            example.createCriteria().andFileIdIn(Arrays.asList(fileIds));

            List<PFile> files = pFileMapper.selectByExample(example);

            Map<String, FileInfo> map = new HashMap<>();
            for (String f : fileIds) {
                map.put(f, files.stream().filter(p -> p.getFileId().equalsIgnoreCase(f)).map(p -> transform(p)).findAny().orElse(null));
            }
            return map;
        }
    }

    /**
     * 获取一组文件信息
     *
     * @param fileIds
     * @return
     */
    public List<FileInfo> listInfo(String... fileIds) {
        if (fileIds != null && fileIds.length > 0) {
            PFileExample example = new PFileExample();
            example.createCriteria().andFileIdIn(Arrays.asList(fileIds));
            return pFileMapper.selectByExample(example).stream().map(p -> transform(p)).collect(Collectors.toList());
        } else {
            return new ArrayList<>();
        }
    }

    /**
     * 获取文件数据
     *
     * @param fileId
     * @return
     */
    public byte[] getFile(String fileId) {
        return iFileStorage.get(fileId);
    }

    /**
     * 获取文件流(调用方在使用完成后应手动关闭该流)
     *
     * @param fileId
     * @return
     */
    public InputStream getFileStream(String fileId) {
        return iFileStorage.getInputStream(fileId);
    }


    public FileInfo transform(PFile file) {
        if (file == null) {
            return null;
        }
        FileInfo info = new FileInfo();
        info.setFileSize(file.getFileSize());
        info.setGmtCreate(file.getGmtCreate());
        info.setFileMime(file.getFileMime());
        info.setFileId(file.getFileId());
        info.setInputUser(file.getInputUser());
        info.setOriginalName(file.getOriginalName());

        return info;
    }
}
