package com.help.service;

import com.help.common.ICacheable;
import com.help.domain.DicItem;
import com.help.domain.TreeDicItem;
import com.help.constant.CacheConstant;
import com.help.constraint.IDicExtension;
import com.help.constraint.ITreeDicExtension;
import com.help.storage.IDicStorage;
import com.help.storage.ITreeDicStorage;
import org.springframework.aop.framework.AopContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.CacheManager;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;

/**
 * 系统字典查询业务类(缓存)
 */
@Service
public class DictionaryService implements ICacheable {

    @Autowired(required = false)
    private List<IDicExtension> iDicExtensions;
    @Autowired(required = false)
    private List<ITreeDicExtension> iTreeDicExtensions;
    @Autowired(required = false)
    private IDicStorage iDicStorage;
    @Autowired(required = false)
    private ITreeDicStorage iTreeDicStorage;

    @Autowired(required = false)
    CacheManager cacheManager;

    /**
     * 通过字典类型获取缓存的字典数据(包括扩展字典)
     *
     * @param dicType
     * @return
     */
    public List<DicItem> list(String dicType) {
        if (iDicExtensions != null) {
            IDicExtension iDicExtension = iDicExtensions.stream().filter(p -> p.getDicType().equalsIgnoreCase(dicType)).findFirst().orElse(null);
            if (iDicExtension != null) {
                if (iDicExtension.cacheable()) {
                    return ((DictionaryService) AopContext.currentProxy()).listExtensionCache(dicType);
                } else {
                    return iDicExtension.list();
                }
            }
        }
        return ((DictionaryService) AopContext.currentProxy()).listBase(dicType);
    }

    @Cacheable(value = CacheConstant.CACHE_DICTIONARY)
    public List<DicItem> listExtensionCache(String dicType) {
        IDicExtension iDicExtension = iDicExtensions.stream().filter(p -> p.getDicType().equalsIgnoreCase(dicType)).findFirst().orElse(null);
        if (iDicExtension != null) {
            return iDicExtension.list();
        } else {
            return new ArrayList<>();
        }
    }

    /**
     * 通过字典类型获取基本字典数据(不包括扩展字典)
     *
     * @param dicType
     * @return
     */
    @Cacheable(value = CacheConstant.CACHE_DICTIONARY)
    public List<DicItem> listBase(String dicType) {
        if (iDicStorage != null) {
            return iDicStorage.list(dicType);
        } else {
            return new ArrayList<>();
        }
    }

    /**
     * 通过树形字典类型获取缓存的树形字典数据(包括扩展字典)
     *
     * @return
     */
    public List<TreeDicItem> listTree(String dicType) {
        if (iTreeDicExtensions != null) {
            ITreeDicExtension iTreeDicExtension = iTreeDicExtensions.stream().filter(p -> p.getDicType().equalsIgnoreCase(dicType)).findFirst().orElse(null);
            if (iTreeDicExtension != null) {
                if (iTreeDicExtension.cacheable()) {
                    return ((DictionaryService) AopContext.currentProxy()).listTreeExtensionCache(dicType);
                } else {
                    return iTreeDicExtension.list();
                }
            }
        }
        return ((DictionaryService) AopContext.currentProxy()).listTreeBase(dicType);
    }

    /**
     * 通过树形字典类型获取基本缓存的树形字典数据(不包括扩展字典)
     *
     * @param dicType
     * @return
     */
    @Cacheable(value = CacheConstant.CACHE_DICTIONARY)
    public List<TreeDicItem> listTreeBase(String dicType) {
        if (iTreeDicStorage != null) {
            return iTreeDicStorage.list(dicType);
        } else {
            return new ArrayList<>();
        }
//        PTreeDicExample example = new PTreeDicExample();
//        example.createCriteria().andDicTypeEqualTo(dicType);
//
//        return pTreeDicMapper.selectByExample(example).stream().map(p -> new TreeDicItem(p.getCode(), p.getText(), p.getParentCode())).collect(Collectors.toList());
    }

    @Cacheable(value = CacheConstant.CACHE_DICTIONARY)
    public List<TreeDicItem> listTreeExtensionCache(String dicType) {
        ITreeDicExtension iTreeDicExtension = iTreeDicExtensions.stream().filter(p -> p.getDicType().equalsIgnoreCase(dicType)).findFirst().orElse(null);
        if (iTreeDicExtension != null) {
            return iTreeDicExtension.list();
        } else {
            return new ArrayList<>();
        }
    }

    @Override
    public void refresh() {
        if (cacheManager != null) {
            cacheManager.getCache(CacheConstant.CACHE_DICTIONARY).clear();
        }
    }
}