package com.help.plugins;

import com.help.ITokenContext;
import com.help.LoginInfo;
import com.help.common.util.StringUtil;
import com.help.config.HelpManageConfig;
import com.help.constant.CacheConstant;
import com.help.constraint.IDicExtension;
import com.help.constraint.IHelpSystemComponent;
import com.help.domain.DicItem;
import com.help.storage.IUserStorage;
import org.springframework.aop.framework.AopContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 用户字典扩展
 *
 * @author: YuBin-002726
 * @Date: 2019/1/28 9:06
 */
@Component
public class UserDicExtension implements IDicExtension, IHelpSystemComponent {

    @Autowired(required = false)
    IUserStorage iUserStorage;

    @Override
    public String getDicType() {
        return "USER";
    }

    @Autowired
    HelpManageConfig helpManageConfig;

    @Autowired(required = false)
    ITokenContext<LoginInfo> iTokenContext;

    @Override
    public List<DicItem> list() {
        if (helpManageConfig.isMultiLegalEnable() && iTokenContext != null) {
            LoginInfo loginInfo = iTokenContext.getCurrentToken();
            if (loginInfo != null) {
                return ((UserDicExtension) AopContext.currentProxy()).listByLegal(loginInfo.getLegalPersonality(), false);
            }
        }
        return ((UserDicExtension) AopContext.currentProxy()).listByLegal(null, false);
    }

    @Cacheable(CacheConstant.CACHE_DICTIONARY)
    public List<DicItem> listByLegal(String legal, boolean limitIfEmpty) {
        if (iUserStorage != null) {
            if ((StringUtil.isNotEmpty(legal) || limitIfEmpty)) {
                return iUserStorage.getLegalProxy(legal).list().stream().map(p ->
                        new DicItem(p.getUserNo(), p.getUserName() + "(" + p.getUserNo() + ")", getDicType())
                ).collect(Collectors.toList());
            } else {
                return iUserStorage.getLegalProxy(null).list().stream().map(p ->
                        new DicItem(p.getUserNo(), p.getUserName() + "(" + p.getUserNo() + ")", getDicType())
                ).collect(Collectors.toList());
            }
        } else {
            return new ArrayList<>();
        }
    }

    @Override
    public boolean cacheable() {
        //由自身实现缓存,不允许字典服务缓存
        return false;
    }

    @Override
    public String getName() {
        return "用户字典扩展服务(缓存)";
    }
}
