package com.help.plugins;

import com.help.ITokenContext;
import com.help.LoginInfo;
import com.help.common.util.StringUtil;
import com.help.config.HelpManageConfig;
import com.help.constant.CacheConstant;
import com.help.constraint.IHelpSystemComponent;
import com.help.constraint.ITreeDicExtension;
import com.help.domain.TreeDicItem;
import com.help.storage.legal.ILegalableStorage;
import com.help.storage.IOrgStorage;
import org.springframework.aop.framework.AopContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 机构树字典扩展
 *
 * @author: YuBin-002726
 * @Date: 2019/1/30 14:22
 */
@Component
public class OrgTreeDicExtension implements ITreeDicExtension, IHelpSystemComponent {

    @Autowired(required = false)
    IOrgStorage iOrgStorage;

    @Autowired(required = false)
    ITokenContext<LoginInfo> iTokenContext;

    @Autowired
    HelpManageConfig helpManageConfig;

    @Override
    public String getName() {
        return "机构树型字典扩展服务(缓存)";
    }

    @Override
    public String getDicType() {
        return "ORG";
    }

    @Override
    public List<TreeDicItem> list() {
        if (helpManageConfig.isMultiLegalEnable() && iTokenContext != null) {
            LoginInfo loginInfo = iTokenContext.getCurrentToken();
            if (loginInfo != null) {
                return ((OrgTreeDicExtension) AopContext.currentProxy()).listByLegal(loginInfo.getLegalPersonality(), false);
            }
        }
        return ((OrgTreeDicExtension) AopContext.currentProxy()).listByLegal(null, false);
    }

    @Cacheable(CacheConstant.CACHE_DICTIONARY)
    public List<TreeDicItem> listByLegal(String legal, boolean limitIfEmpty) {
        if (iOrgStorage != null) {
            if ((StringUtil.isNotEmpty(legal) || limitIfEmpty) && iOrgStorage instanceof ILegalableStorage) {
                return iOrgStorage.getLegalProxy(legal).listAll().stream().map(p -> new TreeDicItem(p.getOrgNo(), p.getOrgShortName(), p.getParentNo())).collect(Collectors.toList());
            } else {
                return iOrgStorage.getLegalProxy(null).listAll().stream().map(p -> new TreeDicItem(p.getOrgNo(), p.getOrgShortName(), p.getParentNo())).collect(Collectors.toList());
            }
        } else {
            return new ArrayList<>();
        }
    }

    @Override
    public boolean cacheable() {
        //由自身实现缓存,不允许字典服务缓存
        return false;
    }
}
