package com.help;

import com.help.common.IInvocationData;
import com.help.common.UnifyPageData;
import com.help.constraint.IHelpDomain;
import com.help.constraint.IHelpExample;
import com.help.constraint.ISearchable;
import com.help.common.exception.UnifyException;
import com.help.common.util.BeanConvert;
import com.help.common.util.Convert;
import com.help.common.util.StringUtil;
import com.help.common.util.intf.IAction3;
import com.help.common.util.reflect.ReflectUtil;
import com.help.domain.DicItem;
import com.help.domain.TreeDicItem;
import com.help.service.DictionaryService;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;

/**
 * HELP平台后端字段转码器
 *
 * @author YuBin-002726
 */
public class Transcoder implements IInvocationData {

    private Map<String, ISearchable<?, IHelpExample<?, ?>>> services = null;            //以Domain类的完全限定名为Key,Service类为Value
    private Map<String, Class<?>> examples = null;                                        //以Domain类的完全限定名为Key,Method的参数类型为Value
    private DictionaryService dictionaryService;

    private Object entity;
    private List<Map<String, Object>> data;
    private boolean singleMode;
    private boolean replaceMode;

    protected Transcoder(Object entity, boolean replaceMode) {
        this.replaceMode = replaceMode;
        this.entity = entity;
        if (entity != null) {
            if (entity instanceof UnifyPageData<?>) {
                data = BeanConvert.toMapList(((UnifyPageData<?>) entity).getList());
            } else if (entity instanceof Iterable<?>) {
                data = BeanConvert.toMapList((Iterable<?>) entity);
            } else if (entity.getClass().isArray()) {
                data = BeanConvert.toMapList(entity);
            } else {
                singleMode = true;
                data = new LinkedList<>();
                data.add(BeanConvert.toMap(entity));
            }
        }
    }

    public void init(Map<String, ISearchable<?, IHelpExample<?, ?>>> services, Map<String, Class<?>> examples, DictionaryService dictionaryService) {
        this.services = services;
        this.examples = examples;
        this.dictionaryService = dictionaryService;
    }

    /**
     * 设置替换模式
     *
     * @param replaceMode 替换模式 若为true 则将转码后的字段值替换原字段 否则在数据中新增"原字段_displayname"字段作为转码后的值字段
     * @return 修改替换模式后的转换器
     */
    public Transcoder setReplaceMode(boolean replaceMode) {
        this.replaceMode = replaceMode;
        return this;
    }

    /**
     * 将实体bean或bean列表中的每个对象的某个字段使用集合中的对象的对应字段匹配，并转码为集合中对象的另一个字段，然后以指定的转码后字段名插入原集合中
     * 此方法进行值匹配时会将字段值取toString再比较 而不是直接equals
     *
     * @param field               要转码的实体bean中的字段
     * @param list                转码时使用的匹配集合 支持实体集合和Map集合
     * @param matchField          匹配集合中对象的用以匹配实体bean中给定字段的字段名
     * @param displayField        匹配集合中对象的用以加入实体bean的字段名
     * @param transcodedFieldName 转码后的字段名称(若不提供则根据当前的replaceMode来确定替换原字段还是新增_displayname字段)
     *                            例：若beanList中的每个对象bean中含有字段userid和age，现有一个List<User>集合list, User中含有id和name字段，则调用transcode("userid",list,"id","name","userName")的结果为
     *                            含有{userid,age,userName}三个字段的List<Map>,其中userName的值为list中id为bean.userid的对象的name值
     * @return
     */
    public Transcoder transcode(String field, Iterable<?> list, String matchField, String displayField, String transcodedFieldName) {
        if (data != null && data.size() > 0 && list != null) {
            if (data.get(0).containsKey(field)) {
                Iterator<?> listIterator = list.iterator();
                if (listIterator.hasNext()) {
                    Object l = listIterator.next();
                    if (l instanceof Map<?, ?>) {
                        for (Map<String, Object> map : data) {
                            Object value = map.get(field);
                            if (value != null) {
                                for (Object e : list) {
                                    Map<String, Object> obj = (Map<String, Object>) e;
                                    Object matchValue = obj.get(matchField);
                                    if (matchValue != null && matchValue.toString().equals(value.toString())) {
                                        if (StringUtil.isNotEmpty(transcodedFieldName)) {
                                            map.put(transcodedFieldName, obj.get(displayField));
                                        } else if (replaceMode) {
                                            map.put(field, obj.get(displayField));
                                        } else {
                                            map.put(field + "_displayname", obj.get(displayField));
                                        }
                                        break;
                                    }
                                }
                            }
                        }
                    } else {
                        Field mField = ReflectUtil.getField(l.getClass(), matchField);
                        if (mField != null) {
                            Field nField = ReflectUtil.getField(l.getClass(), displayField);
                            if (nField != null) {
                                if (!mField.isAccessible()) {
                                    mField.setAccessible(true);
                                }
                                if (!nField.isAccessible()) {
                                    nField.setAccessible(true);
                                }
                                for (Map<String, Object> map : data) {
                                    Object value = map.get(field);
                                    if (value != null) {
                                        for (Object obj : list) {
                                            try {
                                                Object matchValue = mField.get(obj);
                                                if (matchValue != null && matchValue.toString().equals(value.toString())) {
                                                    if (StringUtil.isNotEmpty(transcodedFieldName)) {
                                                        map.put(transcodedFieldName, nField.get(obj));
                                                    } else if (replaceMode) {
                                                        map.put(field, nField.get(obj));
                                                    } else {
                                                        map.put(field + "_displayname", nField.get(obj));
                                                    }

                                                    break;
                                                }
                                            } catch (Exception e) {
                                                throw new UnifyException("未知原因导致的转码失败", e);
                                            }
                                        }
                                    }
                                }
                            } else {
                                throw new UnifyException("对象[" + l + "]中不存在字段[" + displayField + "]");
                            }
                        } else {
                            throw new UnifyException("对象[" + l + "]中不存在字段[" + matchField + "]");
                        }
                    }
                }
            } else {
                throw new UnifyException("对象[" + entity + "]中不存在字段[" + field + "]");
            }
        }
        for (Map<String, Object> e : data) {
            if (StringUtil.isNotEmpty(transcodedFieldName)) {
                if (!e.containsKey(transcodedFieldName)) {
                    e.put(transcodedFieldName, null);
                }
            } else {
                if (replaceMode && !e.containsKey(field)) {
                    e.put(field, null);
                } else if (!replaceMode && !e.containsKey(field + "_displayname")) {
                    e.put(field + "_displayname", null);
                }
            }
        }
        return this;
    }

    /**
     * 将实体bean或bean列表中的每个对象的某个字段使用集合中的对象的对应字段匹配，并转码为集合中对象的另一个字段，然后以现有的替换模式替换原字段或新增_displayname字段
     * 此方法进行值匹配时会将字段值取toString再比较 而不是直接equals
     *
     * @param field        要转码的实体bean中的字段
     * @param list         转码时使用的匹配集合 支持实体集合和Map集合
     * @param matchField   匹配集合中对象的用以匹配实体bean中给定字段的字段名
     * @param displayField 匹配集合中对象的用以加入实体bean的字段名
     *                     例：若beanList中的每个对象bean中含有字段userid和age，现有一个List<User>集合list, User中含有id和name字段，则调用transcode("userid",list,"id","name")的结果为
     *                     含有{userid,age,userid_displayname}三个字段的List<Map>,其中userid_displayname的值为list中id为bean.userid的对象的name值
     * @return
     */
    public Transcoder transcode(String field, Iterable<?> list, String matchField, String displayField) {
        return transcode(field, list, matchField, displayField, null);
    }

    /**
     * 将实体bean或bean列表中的每个对象的某个字段使用数组中的对象的对应字段匹配，并转码为集合中对象的另一个字段，然后以指定的转码后字段名插入原集合中
     * 此方法进行值匹配时会将字段值取toString再比较 而不是直接equals
     *
     * @param field               要转码的实体bean中的字段
     * @param arr                 转码时使用的匹配数组 支持实体数组和Map数组
     * @param matchField          匹配集合中对象的用以匹配实体bean中给定字段的字段名
     * @param displayField        匹配集合中对象的用以加入实体bean的字段名
     * @param transcodedFieldName 转码后的字段名称(若不提供则根据当前的replaceMode来确定替换原字段还是新增_displayname字段)
     *                            例：若beanList中的每个对象bean中含有字段userid和age，现有一个User[]数组arr, User中含有id和name字段，则调用transcode("userid",arr,"id","name","userName")的结果为
     *                            含有{userid,age,userName}三个字段的List<Map>,其中userName的值为list中id为bean.userid的对象的name值
     * @return
     */
    public <T> Transcoder transcode(String field, T[] arr, String matchField, String displayField, String transcodedFieldName) {
        return transcode(field, Arrays.asList(arr), matchField, displayField, transcodedFieldName);
    }

    /**
     * 将实体bean或bean列表中的每个对象的某个字段使用数组中的对象的对应字段匹配，并转码为集合中对象的另一个字段，然后以指定的转码后字段名插入原集合中
     * 此方法进行值匹配时会将字段值取toString再比较 而不是直接equals
     *
     * @param field        要转码的实体bean中的字段
     * @param arr          转码时使用的匹配数组 支持实体数组和Map数组
     * @param matchField   匹配集合中对象的用以匹配实体bean中给定字段的字段名
     * @param displayField 匹配集合中对象的用以加入实体bean的字段名
     *                     例：若beanList中的每个对象bean中含有字段userid和age，现有一个List<User>集合list, User中含有id和name字段，则调用transcode("userid",list,"id","name")的结果为
     *                     含有{userid,age,userid_displayname}三个字段的List<Map>,其中userid_displayname的值为list中id为bean.userid的对象的name值
     * @return
     */
    public <T> Transcoder transcode(String field, T[] arr, String matchField, String displayField) {
        return transcode(field, Arrays.asList(arr), matchField, displayField, null);
    }

    /**
     * 将实体bean或bean列表中的每个对象的某个字段使用PDic字典/扩展字典(或PTreeDic树形字典/扩展树形字典)数据进行转码，然后以指定的转码后字段名插入原集合中
     * 此方法进行值匹配时会将字段值取toString再比较 而不是直接equals
     *
     * @param field               要转码的实体bean中的字段
     * @param dictype             要匹配的字典(或树形字典)的dictype
     * @param transcodedFieldName 转码后的字段名称(若不提供则根据当前的replaceMode来确定替换原字段还是新增_displayname字段)
     *                            例：若beanList中的每个对象bean中含有字段userid和sex，现有一个dictype为SEX的字典，则调用transcode("sex","SEX","sexText")的结果为
     *                            含有{userid,sex,sexText}三个字段的List<Map>,其中sexText的值为字典SEX中TEXT为User性别的CNNAME值
     * @return
     */
    public Transcoder transcode(String field, String dictype, String transcodedFieldName) {
        return transcode(field, dictype, transcodedFieldName, false);
    }

    /**
     * 将实体bean或bean列表中的每个对象的某个字段使用PDic字典/扩展字典(或PTreeDic树形字典/扩展树形字典)数据进行转码，然后以现有的替换模式替换原字段或新增_displayname字段
     * 此方法进行值匹配时会将字段值取toString再比较 而不是直接equals
     *
     * @param field   要转码的实体bean中的字段
     * @param dictype 要匹配的字典(或树形字典)的dictype
     *                例：若beanList中的每个对象bean中含有字段userid和sex，现有一个dictype为SEX的字典，则调用transcode("sex","SEX")的结果为
     *                含有{userid,sex,sex_displayname}三个字段的List<Map>,其中sex_displayname的值为字典SEX中TEXT为User性别的CNNAME值
     * @return
     */
    public Transcoder transcode(String field, String dictype) {
        return transcode(field, dictype, null, false);
    }

    /**
     * 将实体bean或bean列表中的每个对象的某个字段使用PDic字典/扩展字典(或PTreeDic树形字典/扩展树形字典)数据进行转码，然后以现有的替换模式替换原字段或新增_displayname字段
     * 如果字典为树形字典 此方法可选择对其转码为"父-父-子"的结构 此方法进行值匹配时会将字段值取toString再比较 而不是直接equals
     *
     * @param field      要转码的实体bean中的字段
     * @param dictype    要匹配的字典(或树形字典)的dictype
     * @param withParent 如果该字典类型为树形字典 是否将父级节点的文本加入最终转码结果 若为true 则最终转码结果为"父-子"的结构
     *                   例：若beanList中的每个对象bean中含有字段userid和sex，现有一个dictype为SEX的字典，则调用transcode("sex","SEX")的结果为
     *                   含有{userid,sex,sex_displayname}三个字段的List<Map>,其中sex_displayname的值为字典SEX中TEXT为User性别的CNNAME值
     * @return
     */
    public Transcoder transcode(String field, String dictype, boolean withParent) {
        return transcode(field, dictype, null, withParent);
    }

    /**
     * 将实体bean或bean列表中的每个对象的某个字段使用PDic字典/扩展字典(或PTreeDic树形字典/扩展树形字典)数据进行转码，然后以指定的转码后字段名插入原集合中
     * 如果字典为树形字典 此方法可选择对其转码为"父-父-子"的结构 此方法进行值匹配时会将字段值取toString再比较 而不是直接equals
     *
     * @param field               要转码的实体bean中的字段
     * @param dictype             要匹配的字典(或树形字典)的dictype
     * @param transcodedFieldName 转码后的字段名称(若不提供则根据当前的replaceMode来确定替换原字段还是新增_displayname字段)
     * @param withParent          如果该字典类型为树形字典 是否将父级节点的文本加入最终转码结果 若为true 则最终转码结果为"父-子"的结构
     *                            例：若beanList中的每个对象bean中含有字段userid和sex，现有一个dictype为SEX的字典，则调用transcode("sex","SEX")的结果为
     *                            含有{userid,sex,sex_displayname}三个字段的List<Map>,其中sex_displayname的值为字典SEX中TEXT为User性别的CNNAME值
     * @return
     */
    public Transcoder transcode(String field, String dictype, String transcodedFieldName, boolean withParent) {
        List<DicItem> dics = null;
        if (dictionaryService != null) {
            dics = dictionaryService.list(dictype);
        } else {
            dics = new ArrayList<>();
        }
        if (dics.size() > 0) {
            return transcode(field, dics, "code", "text", transcodedFieldName);
        } else {
            //字典不存在,查询树形字典
            List<TreeDicItem> treeDicItems = null;
            if (dictionaryService != null) {
                treeDicItems = dictionaryService.listTree(dictype);
            } else {
                treeDicItems = new ArrayList<>();
            }

            if (!withParent) {
                return transcode(field, treeDicItems, "code", "text", transcodedFieldName);
            } else {
                List<TreeDicItem> result = new ArrayList<TreeDicItem>();
                IAction3<TreeDicItem, List<TreeDicItem>, List<TreeDicItem>> fetchChild = new IAction3<TreeDicItem, List<TreeDicItem>, List<TreeDicItem>>() {
                    @Override
                    public void execute(TreeDicItem parent, List<TreeDicItem> all, List<TreeDicItem> result) {
                        for (TreeDicItem c : all) {
                            if (StringUtil.isNotEmpty(c.getParent()) && c.getParent().equals(parent.getCode())) {
                                TreeDicItem n = new TreeDicItem(c.getCode(), parent.getText() + "-" + c.getText(), c.getParent());
                                result.add(n);
                                this.execute(n, all, result);
                            }
                        }
                    }
                };
                for (TreeDicItem c : treeDicItems) {
                    if (StringUtil.isEmpty(c.getParent())) {
                        TreeDicItem n = new TreeDicItem(c.getCode(), c.getText(), null);
                        result.add(n);
                        fetchChild.execute(n, treeDicItems, result);
                    }
                }
                return transcode(field, result, "code", "text", transcodedFieldName);
            }
        }
    }

    /**
     * 将实体bean或bean列表中的每个对象的某个字段使用给定类型的数据的对应字段匹配(具体数据有本工具自行查询)，并转码为集合中对象的另一个字段，然后以指定的转码后字段名插入原集合中
     * 此方法进行值匹配时会将字段值取toString再比较 而不是直接equals
     * 特别注意 在使用此方法时 系统的com.help.service包或子包中必须含有以Example为参数且返回List<clazzOfMatch>的@Service注解的系统服务类
     * 例:clazzOfMatch为PUser.class 则系统中必须含有返回public List<PUser> list(PUserExample example)方法的@Service注解的服务类
     *
     * @param field               要转码的实体bean中的字段
     * @param clazzOfMatch        转码时使用的匹配数据的类型(具体数据由本工具自行查询)
     * @param matchField          匹配集合中对象的用以匹配实体bean中给定字段的字段名
     * @param displayField        匹配集合中对象的用以加入实体bean的字段名
     * @param transcodedFieldName 转码后的字段名称(若不提供则根据当前的replaceMode来确定替换原字段还是新增_displayname字段)
     *                            例：若beanList中的每个对象bean中含有字段userid和age，现本系统有表PUser, PUser中含有id和name字段，则调用transcode("userid",PUser.class,"id","name","userName")的结果为
     *                            含有{userid,age,userName}三个字段的List<Map>,其中userName的值为PUser表中id为bean.userid的对象的name值
     * @return
     */
    public <T extends IHelpDomain> Transcoder
    transcode(String field, Class<T> clazzOfMatch, String matchField, String displayField, String transcodedFieldName) {
        List<?> matchList = null;

        List<Object> fields = new ArrayList<Object>();
        if (data != null && data.size() > 0) {
            for (Map<String, Object> entity : data) {
                if (StringUtil.isNotEmpty(entity.get(field))) {
                    if (!fields.contains(entity.get(field))) {
                        fields.add(entity.get(field));
                    }
                }
            }
        }
        if (fields.size() > 0) {
            ISearchable<?, IHelpExample<?, ?>> service = services.get(clazzOfMatch.getName());
            Class<?> exampleClass = examples.get(clazzOfMatch.getName());
            if (service != null && exampleClass != null) {
                try {
                    IHelpExample<T, ?> example = (IHelpExample<T, ?>) exampleClass.newInstance();
                    Object criteria = example.createCriteria();

                    Method andMatchFieldIn = criteria.getClass().getMethod("and" + matchField.substring(0, 1).toUpperCase() + matchField.substring(1) + "In", List.class);
                    List<Object> list = new ArrayList<Object>();

                    Field f = ReflectUtil.getField(clazzOfMatch, matchField);
                    if (f.getType().equals(String.class)) {
                        for (Object v : fields) {
                            list.add(v.toString());
                        }
                    } else if (f.getType().equals(Integer.class) || f.getType().equals(int.class)) {
                        for (Object v : fields) {
                            list.add(Convert.toInt(v));
                        }
                    } else if (f.getType().equals(Long.class) || f.getType().equals(long.class)) {
                        for (Object v : fields) {
                            list.add(Convert.toLong(v));
                        }
                    }
                    ReflectUtil.invoke(criteria, andMatchFieldIn, list);
                    String[] columns = new String[2];
                    columns[0] = matchField;
                    columns[1] = displayField;
                    matchList = service.list(example, columns);
                } catch (Exception e) {
                    throw new UnifyException("通过类[" + clazzOfMatch.getName() + "]构造转码器失败,无法构造[" + clazzOfMatch.getName() + "Example]类，或者找不到[" + matchField + "]字段", e);
                }
            }
        }

        return transcode(field, matchList, matchField, displayField, transcodedFieldName);

    }

    /**
     * 将实体bean或bean列表中的每个对象的某个字段使用给定类型的数据的对应字段匹配(具体数据有本工具自行查询)，并转码为集合中对象的另一个字段，然后以现有的替换模式替换原字段或新增_displayname字段
     * 此方法进行值匹配时会将字段值取toString再比较 而不是直接equals
     * 特别注意 在使用此方法时 系统的com.help.service包或子包中必须含有以Example为参数且返回List<clazzOfMatch>的@Service注解的系统服务类
     * 例:clazzOfMatch为PUser.class 则系统中必须含有返回public List<PUser> list(PUserExample example)方法的@Service注解的服务类
     *
     * @param field        要转码的实体bean中的字段
     * @param clazzOfMatch 转码时使用的匹配数据的类型(具体数据由本工具自行查询)
     * @param matchField   匹配集合中对象的用以匹配实体bean中给定字段的字段名
     * @param displayField 匹配集合中对象的用以加入实体bean的字段名
     *                     例：若beanList中的每个对象bean中含有字段userid和age，现本系统有表PUser, PUser中含有id和name字段，则调用transcode("userid",PUser.class,"id","name")的结果为
     *                     含有{userid,age,userid_displayname}三个字段的List<Map>,其中userid_displayname的值为PUser表中id为bean.userid的对象的name值
     * @return
     */
    public <T extends IHelpDomain> Transcoder transcode(String field, Class<T> clazzOfMatch, String matchField, String displayField) {
        return transcode(field, clazzOfMatch, matchField, displayField, null);
    }

//    /**
//     * 为当前数据集合增加附件相关字段
//     *
//     * @param fileBusiType
//     * @param sourcePkField
//     * @return
//     */
//    public Transcoder linkFile(String fileBusiType, String sourcePkField) {
//        List<PCount> matchList = null;
//
//        List<String> fields = new ArrayList();
//        if ((this.data != null) && (this.data.size() > 0)) {
//            for (Map entity : this.data) {
//                if ((StringUtil.isNotEmpty(entity.get(sourcePkField))) &&
//                        (!fields.contains(entity.get(sourcePkField).toString()))) {
//                    fields.add(entity.get(sourcePkField).toString());
//                }
//            }
//        }
//
//        if (fields.size() > 0) {
//            PBusiFileService storage = SpringContextUtil.getApplicationContext().getBean(PBusiFileService.class);
//            matchList = storage.listCount(fileBusiType, fields);
//            for (String pk : fields) {
//                boolean contains = false;
//                for (PCount c : matchList) {
//                    if (c.getItem().equals(pk)) {
//                        contains = true;
//                        break;
//                    }
//                }
//                if (!contains) {
//                    PCount c = new PCount();
//                    c.setItem(pk);
//                    c.setItemCount(Integer.valueOf(0));
//                    matchList.add(c);
//                }
//            }
//        }
//
//        return transcode(sourcePkField, matchList, "item", "itemCount", "fileCount");
//    }


    /**
     * 获取转码结果
     *
     * @return
     */
    @Override
    public Object getResult() {
        if (entity == null) {
            return null;
        } else if (entity instanceof UnifyPageData<?>) {
            UnifyPageData<?> unifyPageData = (UnifyPageData<?>) entity;
            return new UnifyPageData<>(data, unifyPageData.getPageIndex(), unifyPageData.getPageSize(), unifyPageData.getTotalSize());
        } else if (data == null) {
            return null;
        } else if (singleMode) {
            return data.get(0);
        } else {
            return data;
        }
    }
}