package com.help;

import com.help.annotation.UnifyAuthorization;
import com.help.common.util.StringUtil;
import com.help.common.util.reflect.ReflectUtil;
import com.help.constant.CommonConstant;
import com.help.constant.PermissionConstant;
import com.help.web.module.Module;
import com.help.web.module.Operation;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.PostConstruct;
import java.lang.reflect.Method;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 模块解析器
 * 本组件会自动排除在当前系统无效的功能
 *
 * @author: YuBin-002726
 * @Date: 2019/3/5 18:01
 */
@Component
public class ModuleAnalasys {

    @Autowired
    ApplicationContext applicationContext;

    //所有模块
    private Map<String, Module> modules;
    //所有所有用户永久具有的操作权限
    private List<Operation> grantToAll;

    @PostConstruct
    public void init() {
        List<Object> ctrls = new ArrayList<>();
        ctrls.addAll(applicationContext.getBeansWithAnnotation(Controller.class).values());
        ctrls.addAll(applicationContext.getBeansWithAnnotation(RestController.class).values());

        Map<String, Module> modules = new HashMap<>();
        List<Operation> grantToAll = new ArrayList<>();

        for (Object o : ctrls) {
            Class<?> clz = AopUtils.getTargetClass(o);
            UnifyAuthorization classAnno = clz.getAnnotation(UnifyAuthorization.class);
            List<Method> mtds = ReflectUtil.getAllMethods(clz);
            for (Method m : mtds) {
                UnifyAuthorization anno = m.getAnnotation(UnifyAuthorization.class);
                if (classAnno != null || anno != null) {
                    String module = null;
                    String op = null;
                    String system = null;
                    String moduleName = null;
                    String opName = null;
                    boolean all = false;

                    if (anno != null) {
                        module = anno.module();
                        moduleName = anno.moduleName();
                        op = anno.op();
                        opName = anno.opName();
                        system = anno.system();
                        all = anno.grantToAll();
                    }

                    if (classAnno != null) {
                        if (StringUtil.isEmpty(module)) {
                            module = classAnno.module();
                            moduleName = classAnno.moduleName();
                        }
                        if (StringUtil.isEmpty(op)) {
                            op = classAnno.op();
                            opName = classAnno.opName();
                        }
                        if (StringUtil.isEmpty(system)) {
                            system = classAnno.system();
                        }
                        if (anno == null) {
                            all = classAnno.grantToAll();
                        }
                    }

                    if (StringUtil.isNotEmpty(module)) {
                        Module md = modules.get(module);
                        if (md == null) {
                            md = new Module();
                            md.setModule(module);
                            md.setName(StringUtil.nvl(moduleName, module));
                            md.setSystem(CommonConstant.YES.equalsIgnoreCase(system) ? "1" : "0");
                            modules.put(module, md);
                        }
                        if (StringUtil.isNotEmpty(op)) {
                            Operation operation = new Operation();
                            operation.setModule(module);
                            operation.setOp(op);
                            operation.setName(StringUtil.nvl(opName, transOp(op)));
                            md.addOperation(operation);

                            if (all) {
                                operation.setReadonly(true);
                                grantToAll.add(operation);
                            }
                        }
                    }
                }

            }
        }

        for (Module m : modules.values()) {
            Iterator<Operation> it = m.getOperations().iterator();
            Operation find = null;
            while (it.hasNext()) {
                Operation op = it.next();
                if (op.getOp().equalsIgnoreCase(PermissionConstant.PERMISSION_VISIT)) {
                    it.remove();
                    find = op;
                }
            }
            if (find != null) {
                m.getOperations().add(0, find);
            }
        }
        this.modules = modules;
        this.grantToAll = grantToAll;

    }

    /**
     * 列出所有功能
     *
     * @return
     */
    public List<Operation> listOps() {
        List<Operation> ops = new ArrayList<>();
        for (Module m : modules.values()) {
            ops.addAll(m.getOperations());
        }
        return ops;
    }

    /**
     * 列出所有模块
     *
     * @return
     */
    public List<Module> list() {
        return new ArrayList<>(modules.values());
    }

    /**
     * 列出所有系统功能
     *
     * @return
     */
    public List<Operation> listSystemOps() {
        List<Operation> ops = new ArrayList<>();
        listSystem().stream().forEach(p -> {
            ops.addAll(p.getOperations());
        });
        return ops;
    }

    /**
     * 列出所有系统模块
     *
     * @return
     */
    public List<Module> listSystem() {
        return modules.values().stream().filter(p -> CommonConstant.YES.equalsIgnoreCase(p.getSystem())).collect(Collectors.toList());
    }

    /**
     * 列出所有默认授权给所有用户的权限
     *
     * @return
     */
    public List<Operation> listGrantToAll() {
        return grantToAll;
    }

    public Module get(String module) {
        return modules.get(module);
    }

    private String transOp(String op) {
        if (PermissionConstant.PERMISSION_ADD.equalsIgnoreCase(op)) {
            return "新增";
        } else if (PermissionConstant.PERMISSION_UPDATE.equalsIgnoreCase(op)) {
            return "修改";
        } else if (PermissionConstant.PERMISSION_DELETE.equalsIgnoreCase(op)) {
            return "删除";
        } else if (PermissionConstant.PERMISSION_DETAIL.equalsIgnoreCase(op)) {
            return "详情";
        } else if (PermissionConstant.PERMISSION_VISIT.equalsIgnoreCase(op)) {
            return "基本查询";
        } else {
            return op;
        }
    }

}
