package com.help.aop;

import com.help.common.HelpIapMessageBuilder;
import com.help.common.exception.UnifyErrorCode;
import com.help.common.exception.UnifyException;
import com.help.common.exception.UnifyValidateException;
import com.help.common.util.StringUtil;
import com.help.constraint.ISerNoGenerator;
import com.help.crcb.hub.CrcbHubRequestInfo;
import com.help.crcb.hub.CrcbHubResponseInfo;
import com.help.iap.HelpIapRequestInfo;
import com.help.iap.HelpIapResponseInfo;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.Ordered;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.DuplicateKeyException;

import java.util.concurrent.ExecutionException;

/**
 * 返回HelpIapResponseInfo的HTTP协议调用全局异常处理器
 *
 * @author JBF-YB
 */
@Aspect
public class HttpIapInvokeExceptionGlobalAspect implements Ordered {

    private Logger logger = LoggerFactory.getLogger(HttpIapInvokeExceptionGlobalAspect.class);

    public HttpIapInvokeExceptionGlobalAspect(HelpIapMessageBuilder helpIapMessageBuilder, ISerNoGenerator iSerNoGenerator) {
        this.helpIapMessageBuilder = helpIapMessageBuilder;
        this.iSerNoGenerator = iSerNoGenerator;
    }

    HelpIapMessageBuilder helpIapMessageBuilder;

    ISerNoGenerator iSerNoGenerator;

    @Around("(@within(org.springframework.stereotype.Controller) || @within(org.springframework.web.bind.annotation.RestController) ) && execution(public com.help.iap.HelpIapResponseInfo *(..))")
    public Object executeOld(ProceedingJoinPoint pjp) throws Throwable {
        try {
            return pjp.proceed();
        } catch (Exception e) {
            CrcbHubResponseInfo responseInfo = handle(e, getRequest(pjp));
            HelpIapResponseInfo rsp = new HelpIapResponseInfo();
            rsp.setResponseHead(responseInfo.getResponseHead());

            return rsp;
        }
    }

    @Around("(@within(org.springframework.stereotype.Controller) || @within(org.springframework.web.bind.annotation.RestController) ) && execution(public com.help.crcb.hub.CrcbHubResponseInfo *(..))")
    public Object execute(ProceedingJoinPoint pjp) throws Throwable {
        try {
            return pjp.proceed();
        } catch (Exception e) {
            return handle(e, getRequest(pjp));
        }
    }


    private CrcbHubResponseInfo handle(Throwable exp, CrcbHubRequestInfo requestInfo) {
        if (exp instanceof org.springframework.validation.BindException) {
            CrcbHubResponseInfo responseInfo = helpIapMessageBuilder.buildHubResponse(requestInfo.getRequestHead());
            responseInfo.setError(UnifyErrorCode.VALIDATE_FAILD);
            if (iSerNoGenerator != null) {
                responseInfo.getResponseHead().setSplrSeqNo(iSerNoGenerator.getRandomSerno());
            }
            return responseInfo;
        } else if (exp instanceof UnifyValidateException) {
            UnifyValidateException e = (UnifyValidateException) exp;
            if (e.getError() != null) {
                logger.info("接口调用参数验证失败,错误参数[" + StringUtil.join(e.getError().keySet(), ",") + "]");
            } else {
                logger.info("接口参数验证失败:[" + e.getMessage() + "]", e);
            }
            CrcbHubResponseInfo responseInfo = helpIapMessageBuilder.buildHubResponse(requestInfo.getRequestHead());
            responseInfo.setError(e);
            if (iSerNoGenerator != null) {
                responseInfo.getResponseHead().setSplrSeqNo(iSerNoGenerator.getRandomSerno());
            }
            return responseInfo;
        } else if (exp instanceof UnifyException) {
            UnifyException e = (UnifyException) exp;
            if (e.getErrorCode().getLevel() < 3) {
                logger.info("接口调用失败:[" + e.getMessage() + "]", e);
            } else if (e.getErrorCode().getLevel() < 4) {
                logger.info("接口调用失败:[" + e.getMessage() + "]", e);
            } else {
                logger.error("接口调用失败:[" + e.getMessage() + "]", e);
            }
            CrcbHubResponseInfo responseInfo = helpIapMessageBuilder.buildHubResponse(requestInfo.getRequestHead());
            responseInfo.setError(e);
            if (iSerNoGenerator != null) {
                responseInfo.getResponseHead().setSplrSeqNo(iSerNoGenerator.getRandomSerno());
            }
            return responseInfo;
        } else if (exp instanceof DuplicateKeyException) {
            DuplicateKeyException e = (DuplicateKeyException) exp;
            logger.warn("数据库主键冲突异常:[" + e.getMessage() + "]", e);
            CrcbHubResponseInfo responseInfo = helpIapMessageBuilder.buildHubResponse(requestInfo.getRequestHead());
            responseInfo.setError(UnifyErrorCode.BUSINESS_FAILD, "请勿录入重复数据");
            if (iSerNoGenerator != null) {
                responseInfo.getResponseHead().setSplrSeqNo(iSerNoGenerator.getRandomSerno());
            }
            return responseInfo;
        } else if (exp instanceof DataAccessException) {
            DataAccessException e = (DataAccessException) exp;
            logger.warn("数据库操作异常:[" + e.getMessage() + "]", e);
            CrcbHubResponseInfo responseInfo = helpIapMessageBuilder.buildHubResponse(requestInfo.getRequestHead());
            responseInfo.setError(UnifyErrorCode.BUSINESS_FAILD, "数据库操作失败");
            if (iSerNoGenerator != null) {
                responseInfo.getResponseHead().setSplrSeqNo(iSerNoGenerator.getRandomSerno());
            }
            return responseInfo;
        } else if (exp instanceof ExecutionException) {
            return handle(exp.getCause(), requestInfo);
        } else if (exp instanceof Throwable) {
            logger.error("接口调用失败,未知错误:[" + exp.getMessage() + "]", exp);
            CrcbHubResponseInfo responseInfo = helpIapMessageBuilder.buildHubResponse(requestInfo.getRequestHead());
            responseInfo.setError(UnifyErrorCode.UNKNOW_FAIL);
            if (iSerNoGenerator != null) {
                responseInfo.getResponseHead().setSplrSeqNo(iSerNoGenerator.getRandomSerno());
            }
            return responseInfo;
        } else {
            CrcbHubResponseInfo<Void> responseInfo = helpIapMessageBuilder.buildHubResponse(requestInfo.getRequestHead());
            responseInfo.setError(UnifyErrorCode.UNKNOW_FAIL);
            if (iSerNoGenerator != null) {
                responseInfo.getResponseHead().setSplrSeqNo(iSerNoGenerator.getRandomSerno());
            }
            return responseInfo;
        }
    }

    private CrcbHubRequestInfo<Void> getRequest(ProceedingJoinPoint pjp) {
        Object[] args = pjp.getArgs();
        if (args != null) {
            for (int i = 0; i < args.length; i++) {
                if (args[i] instanceof HelpIapRequestInfo) {
                    CrcbHubRequestInfo requestInfo = new CrcbHubRequestInfo();
                    requestInfo.setRequestHead(((HelpIapRequestInfo) args[i]).getRequestHead());
                    return requestInfo;
                } else if (args[i] instanceof CrcbHubRequestInfo) {
                    return (CrcbHubRequestInfo) args[i];
                }
            }
        }
        return new CrcbHubRequestInfo<>();
    }

    @Override
    public int getOrder() {
        return 10;
    }
}
