package fox.base;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

/**
 * 扩展点注册表
 * <p>
 * Created by 江成 on 2016/10/13.
 */

public class ExtensionRegistry {


    /**
     * 扩展点注册表
     */
    private Map<String, IExtensionPoint> extensionPointMap = new HashMap<String, IExtensionPoint>();

    /**
     * 构造函数
     */
    public ExtensionRegistry() {

    }

    /**
     * 加载扩展点配置
     *
     * @param file
     * @param encoding
     *
     * @throws XmlPullParserException,IllegalFormatException,IOException
     */
    public void load(File file,String encoding)  throws XmlPullParserException,IllegalFormatException,IOException {

        InputStream in;
        try{
            in = new BufferedInputStream(new FileInputStream((file)));
            this.load(in,encoding);
        }finally{

        }
    }

    /**
     * 加载扩展点配置
     *
     * @param in
     * @param encoding
     *
     * @throws XmlPullParserException,IllegalFormatException,IOException
     */
    public void load(InputStream in,String encoding)  throws XmlPullParserException,IllegalFormatException,IOException {
        this.parserFile(in, encoding, this.extensionPointMap);
    }


    /**
     * 解析文件
     *
     * @param input
     * @param encoding
     * @param extensionPointMap
     *
     * @throws XmlPullParserException,IllegalFormatException,IOException
     */
    private void parserFile(InputStream input, String encoding, Map<String, IExtensionPoint> extensionPointMap)
            throws XmlPullParserException,IllegalFormatException,IOException {
        //获取xml解析工厂
        XmlPullParserFactory factory = XmlPullParserFactory.newInstance();
        factory.setNamespaceAware(true);
        XmlPullParser parser = factory.newPullParser();
        parser.setInput(input, encoding);

        //获取事件类型
        int eventType = parser.getEventType();

        //判断xml格式是否正确
        if (eventType != XmlPullParser.START_DOCUMENT) {
            throw new IllegalFormatException("Not starting with 'start_document'");
        }

        //堆栈
        Stack<StackNode> stack = new Stack<StackNode>();
        //获取下一个事件
        eventType = parser.next();
        while (eventType != XmlPullParser.END_DOCUMENT) {
            //根据不同的事件，进行处理
            switch (eventType) {
                case XmlPullParser.START_TAG: {
                    //获取tagName
                    String tagName = parser.getName();
                    if ("plugins".equals(tagName)) {
                        //do nothing
                    } else if ("extension".equals(tagName)) {
                        Extension extension = new Extension();
                        //获取属性数量
                        int count = parser.getAttributeCount();
                        String name = "";
                        //设置属性
                        for (int i = 0; i < count; i++) {
                            String key = parser.getAttributeName(i);
                            String value = parser.getAttributeValue(i);
                            if("point".equalsIgnoreCase(key)){
                                name=value;
                            }
                            extension.setAttribute(key, value);
                        }
                        //定义节点
                        StackNode node = new StackNode();
                        node.name = name;
                        node.obj = extension;
                        //加入堆栈
                        stack.push(node);
                    } else {
                        //获取属性数量
                        int count = parser.getAttributeCount();
                        //定义配置元素
                        ConfigElement configElement = new ConfigElement();
                        //设置名称
                        configElement.setName(tagName);
                        //设置属性
                        for (int i = 0; i < count; i++) {
                            String key = parser.getAttributeName(i);
                            String value = parser.getAttributeValue(i);
                            configElement.setAttribute(key, value);
                        }
                        //定义节点
                        StackNode node = new StackNode();
                        node.name = tagName;
                        node.obj = configElement;
                        //加入堆栈
                        stack.push(node);
                    }
                    break;
                }
                case XmlPullParser.END_TAG: {
                    //获取tagName
                    String tagName = parser.getName();
                    if ("plugins".equals(tagName)) {
                        //do nothing
                    } else if ("extension".equals(tagName)) {
                        // 移除栈顶对象
                        StackNode node = stack.pop();
                        // 获取扩展
                        Extension extension = (Extension) node.obj;
                        // 获取扩展点
                        ExtensionPoint extensionPoint = (ExtensionPoint) extensionPointMap.get(node.name);
                        if (extensionPoint == null) {
                            extensionPoint = new ExtensionPoint();
                            extensionPointMap.put(node.name, extensionPoint);
                        }
                        extensionPoint.addExtension(extension);

                    } else {
                        if(!stack.isEmpty()) {
                            // 移除栈顶对象
                            StackNode node = stack.pop();
                            //查看栈顶对象而不移除它
                            if(!stack.isEmpty()) {
                                StackNode parent = stack.peek();
                                if (parent.obj instanceof Extension) {
                                    Extension extension = (Extension) parent.obj;
                                    ConfigElement configElement = (ConfigElement) node.obj;
                                    extension.addConfigElement(configElement);
                                } else if (parent.obj instanceof IConfigElement) {
                                    ConfigElement parentElement = (ConfigElement) parent.obj;
                                    ConfigElement subElement = (ConfigElement) node.obj;
                                    parentElement.addChildren(node.name, subElement);
                                }
                            }
                        }
                        break;
                    }
                }
                case XmlPullParser.TEXT: {
                    if(!stack.isEmpty()) {
                        //查看栈顶对象而不移除它
                        StackNode node = stack.peek();
                        if (node.obj instanceof IConfigElement) {
                            ConfigElement subElement = (ConfigElement) node.obj;
                            subElement.setText(parser.getText());
                        }
                    }
                    break;
                }
            }
            //跳转下一个事件
            eventType = parser.next();
        }
    }

    /**
     * 获取扩展点集合
     * @param pointName
     * @return
     */
    public IExtensionPoint getExtensionPoint(String pointName){
        //获取扩展点
        IExtensionPoint extensionPoint = this.extensionPointMap.get(pointName);
        return extensionPoint;
    }

    /**
     * stack node
     */
    private class StackNode {

        /**
         * 名称
         */
        public String name;

        /**
         * 引用对象
         */
        public Object obj;

    }

    /**
     * 扩展点
     */
    private class ExtensionPoint implements IExtensionPoint {
        /**
         * 扩展配置队列
         */
        private List<IExtension> children = new ArrayList<IExtension>();

        /**
         * 获取所有的扩展集合
         *
         * @return
         */
        public IExtension[] getExtensions() {
            IExtension[] list = new IExtension[0];
            list = children.toArray(list);
            return list;
        }

        /**
         * 加入配置元素
         *
         * @param extension
         */
        public void addExtension(IExtension extension) {
            children.add(extension);
        }

    }

    /**
     * 扩展
     */
    private class Extension implements IExtension {

        /**
         * 属性集合
         */
        private Map<String, String> attributes = new HashMap<String, String>();

        /**
         * 扩展配置队列
         */
        private List<IConfigElement> children = new ArrayList<IConfigElement>();

        /**
         * 获取所有的配置元素集合
         *
         * @return
         */
        public IConfigElement[] getConfigElements() {
            IConfigElement[] list = new IConfigElement[0];
            list = children.toArray(list);
            return list;
        }

        /**
         * 加入配置元素
         *
         * @param configElement
         */
        public void addConfigElement(IConfigElement configElement) {
            children.add(configElement);
        }


        /**
         * key集合
         * @return
         */
        public List<String> attributeKeys(){
            List<String> keys = new ArrayList<>();
            keys.addAll(this.attributes.keySet());
            return keys;
        }

        /**
         * 获取扩展点属性
         *
         * @param key
         * @return
         */
        public String getAttribute(String key) {
            String value = this.attributes.get(key);
            return value;
        }

        /**
         * 设置属性值
         *
         * @param key
         * @param value
         */
        public void setAttribute(String key, String value) {
            this.attributes.put(key, value);
        }

    }


    /**
     * 扩展配置元素
     */
    private class ConfigElement implements IConfigElement {

        /**
         * 名称
         */
        private String name;

        /**
         * text
         */
        private String text;

        /**
         * 属性集合
         */
        private Map<String, String> attributes = new HashMap<String, String>();

        /**
         * 孩子集合
         */
        private Map<String, List<IConfigElement>> childrenMap = new HashMap<String, List<IConfigElement>>();

        /**
         * key集合
         * @return
         */
        public List<String> attributeKeys(){
            List<String> keys = new ArrayList<>();
            keys.addAll(this.attributes.keySet());
            return keys;
        }

        /**
         * 获取扩展点配置名称
         *
         * @return
         */
        public String getName() {
            return this.name;
        }

        /**
         * 设置扩展点配置名称
         *
         * @param name
         */
        public void setName(String name) {
            this.name = name;
        }

        /**
         * 获取扩展点的文本
         *
         * @return
         */
        public String getText() {
            return this.text;
        }

        /**
         * 设置扩展点的文本
         *
         * @param text
         */
        public void setText(String text) {
            this.text = text;
        }

        /**
         * 获取扩展点属性
         *
         * @param key
         * @return
         */
        public String getAttribute(String key) {
            String value = this.attributes.get(key);
            return value;
        }

        /**
         * 设置属性值
         *
         * @param key
         * @param value
         */
        public void setAttribute(String key, String value) {
            this.attributes.put(key, value);
        }

        /**
         * 获取指定名称的扩展点配置列表
         *
         * @param name
         * @return
         */
        public IConfigElement[] getChildren(String name) {
            List<IConfigElement> list = this.childrenMap.get(name);
            if (list == null) {
                return new IConfigElement[0];
            }
            IConfigElement[] children = list.toArray(new IConfigElement[0]);
            return children;
        }

        /**
         * 加入扩展点配置列表
         *
         * @param name
         * @param configElement
         */
        public void addChildren(String name, IConfigElement configElement) {
            List<IConfigElement> list = this.childrenMap.get(name);
            if (list == null) {
                list = new ArrayList<IConfigElement>();
                this.childrenMap.put(name, list);
            }
            //加入队列
            list.add(configElement);
        }


    }


}
