/*
 * Decompiled with CFR 0.152.
 */
package cn.com.yusys.yusp.commons.util.io;

import cn.com.yusys.yusp.commons.util.StringUtils;
import cn.com.yusys.yusp.commons.util.io.IOUtils;
import java.io.BufferedInputStream;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtils {
    public static final String FILE_EXISTS_BUT_IS_DIRECTORY = "File '%s' exists but is a directory";
    public static final String FILE_CANNOT_BE_READ = "File '%s' cannot be read";
    public static final String FILE_DOES_NOT_EXIST = "File '%s' does not exist";
    public static final String FILE_CANNOT_BE_WRITTEN = "File '%s' cannot be written to";
    public static final String DIRECTORY_COULD_NOT_BE_CREATED = "Directory '%s' could not be created";
    private static final Logger log = LoggerFactory.getLogger(FileUtils.class);
    private static final int DEFAULT_BUFFER_SIZE = 4096;

    private FileUtils() {
    }

    public static FileInputStream openInputStream(File file) throws IOException {
        if (file.exists()) {
            if (file.isDirectory()) {
                throw new IOException(String.format(FILE_EXISTS_BUT_IS_DIRECTORY, file));
            }
            if (!file.canRead()) {
                throw new IOException(String.format(FILE_CANNOT_BE_READ, file));
            }
        } else {
            throw new FileNotFoundException(String.format(FILE_DOES_NOT_EXIST, file));
        }
        return new FileInputStream(file);
    }

    public static FileOutputStream openOutputStream(File file) throws IOException {
        return FileUtils.openOutputStream(file, false);
    }

    public static FileOutputStream openOutputStream(File file, boolean append) throws IOException {
        if (file.exists()) {
            if (file.isDirectory()) {
                throw new IOException(String.format(FILE_EXISTS_BUT_IS_DIRECTORY, file));
            }
            if (!file.canWrite()) {
                throw new IOException(String.format(FILE_CANNOT_BE_WRITTEN, file));
            }
        } else {
            File parent = file.getParentFile();
            if (parent != null && !parent.mkdirs() && !parent.isDirectory()) {
                throw new IOException(String.format(DIRECTORY_COULD_NOT_BE_CREATED, parent));
            }
        }
        return new FileOutputStream(file, append);
    }

    public static long copyStream(InputStream input, OutputStream output) throws IOException {
        byte[] buffer = new byte[4096];
        long count = 0L;
        int n = 0;
        while (-1 != (n = input.read(buffer))) {
            output.write(buffer, 0, n);
            count += (long)n;
        }
        return count;
    }

    public static File openFileDir(String dirPath) throws IOException {
        File dirFile = new File(dirPath);
        if (!dirFile.exists()) {
            if (!dirFile.mkdirs() && !dirFile.isDirectory()) {
                throw new IOException("Directory '" + dirFile + "' could not be created");
            }
        } else if (!dirFile.isDirectory()) {
            throw new IOException("File '" + dirFile + "' exists but is not a directory");
        }
        return dirFile;
    }

    public static String readAll(String filePath) throws IOException {
        return FileUtils.readAll(filePath, StandardCharsets.UTF_8.name());
    }

    public static String readAll(String filePath, String charsetName) throws IOException {
        return FileUtils.readAllLines(filePath, charsetName).stream().reduce("", String::concat);
    }

    public static List<String> readAllLines(String filePath) throws IOException {
        return FileUtils.readAllLines(filePath, StandardCharsets.UTF_8.name());
    }

    public static List<String> readAllLines(String filePath, String charsetName) throws IOException {
        Path path = Paths.get(filePath, new String[0]);
        if (Files.exists(path, new LinkOption[0]) && !Files.isDirectory(path, new LinkOption[0]) && path.toFile().length() > 0L) {
            return Files.readAllLines(path, Charset.forName(charsetName));
        }
        return new ArrayList<String>();
    }

    public static void createFile(String filePath) throws IOException {
        FileUtils.createFile(Paths.get(filePath, new String[0]));
    }

    public static void createFile(Path path) throws IOException {
        if (Objects.isNull(path)) {
            throw new IOException("\u6587\u4ef6\u521b\u5efa\u5931\u8d25\uff0c\u5f53\u524dPath\u4e3a\u7a7a!");
        }
        if (Files.isDirectory(path, new LinkOption[0])) {
            throw new IOException("\u6587\u4ef6\u521b\u5efa\u5931\u8d25\uff0c\u5f53\u524d\u8def\u5f84\uff1a" + path.toString() + "\u4e3a\u6587\u4ef6\u5939!");
        }
        Files.deleteIfExists(path);
        path.toFile().getParentFile().mkdirs();
        Files.createFile(path, new FileAttribute[0]);
    }

    public static void fileCopy(String srcFilePath, String targetFilePath) throws IOException {
        Files.copy(Paths.get(srcFilePath, new String[0]), Paths.get(targetFilePath, new String[0]), new CopyOption[0]);
    }

    public static void fileCoverCopy(String srcFilePath, String targetFilePath) throws IOException {
        Path targetPath = Paths.get(targetFilePath, new String[0]);
        Files.deleteIfExists(targetPath);
        Files.copy(Paths.get(srcFilePath, new String[0]), targetPath, new CopyOption[0]);
    }

    public static String byte2File(byte[] bytes, String filePath) throws IOException {
        Path path = Paths.get(filePath, new String[0]);
        if (FileUtils.isFile(filePath)) {
            FileUtils.createFile(path);
            Files.write(path, bytes, new OpenOption[0]);
        }
        return path.toAbsolutePath().toString();
    }

    public static String byte2File(byte[] bytes, String filePath, String charset) throws IOException {
        Path path = Paths.get(filePath, new String[0]);
        if (FileUtils.isFile(filePath)) {
            FileUtils.createFile(path);
            try (BufferedWriter bw = Files.newBufferedWriter(path, Charset.forName(charset), new OpenOption[0]);){
                bw.write(new String(bytes, charset));
            }
            catch (Exception e) {
                log.error("Write bytes to file failed.", (Throwable)e);
            }
        }
        return path.toAbsolutePath().toString();
    }

    public static boolean isFile(String filePath) {
        return !FileUtils.isDirectory(filePath);
    }

    public static boolean isDirectory(String filePath) {
        return Files.isDirectory(Paths.get(filePath, new String[0]), new LinkOption[0]);
    }

    public static String write(Collection<String> lineTexts, String filePath, String charset) throws IOException {
        return FileUtils.write(lineTexts, filePath, charset, false);
    }

    public static String write(Collection<String> lineTexts, String filePath, String charset, boolean isAppend) throws IOException {
        if (StringUtils.isEmpty(charset)) {
            charset = Charset.defaultCharset().name();
        }
        Path path = Paths.get(filePath, new String[0]);
        if (FileUtils.isFile(filePath)) {
            if (!Files.exists(path, new LinkOption[0])) {
                FileUtils.createFile(path.toAbsolutePath().toString());
            }
            OpenOption[] openOptions = null;
            openOptions = isAppend ? new OpenOption[]{StandardOpenOption.WRITE, StandardOpenOption.APPEND, StandardOpenOption.SYNC} : new OpenOption[]{};
            Files.write(path, lineTexts, Charset.forName(charset), openOptions);
        }
        return path.toAbsolutePath().toString();
    }

    public static BufferedInputStream getInputStream(File file) throws RuntimeException {
        try {
            return IOUtils.toBuffered(IOUtils.toStream(file));
        }
        catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    public static BufferedInputStream getInputStream(String path) throws RuntimeException {
        return FileUtils.getInputStream(new File(path));
    }

    public static boolean deleteFileIfExists(File file) {
        try {
            return Files.deleteIfExists(file.toPath());
        }
        catch (IOException e) {
            log.warn("delete File[{}] failure,cause by:{}.", (Object)file, (Object)e.getMessage());
            return false;
        }
    }

    public static void transfer(File srcFile, File targetFile) throws IOException {
        try (FileInputStream inStream = new FileInputStream(srcFile);
             FileOutputStream outStream = new FileOutputStream(targetFile);
             FileChannel in = inStream.getChannel();
             FileChannel out = outStream.getChannel();){
            in.transferTo(0L, in.size(), out);
        }
    }

    public static String trimSeparatorSuffix(String path) {
        if (path.endsWith(File.separator)) {
            path = path.substring(0, path.length() - File.separator.length());
        }
        return path;
    }
}

